﻿using System;
using System.Windows.Forms;
using System.IO;
using XdsObjects;
using XdsObjects.Enums;

namespace XDS_Repository__Simple_Version_
{
    public partial class RepositoryForm : Form
    {
        public RepositoryForm()
        {
            InitializeComponent();
        }

        #region "Varibles and Constants"
        // the Server object which handles the incoming Provide-and-Register and Retrieve requests
        XdsMtomServer server;
        XdsDomain xds;

        // The registry address - where we register the received documents
        const string registryURI = "http://localhost:8090/MyRegistry";
        // The UniqueID of this Repository
        const string MyUID = "1.19.6.24.109.42.1.5";
        // There the received documents get stored
        const string StoragePath = @"C:\XDSSampleRepositoryData\";
        // The web address this Repository can be 
        const string MyURI = "http://localhost:8080/MyRepository";
        #endregion

        public delegate void LogMessageHandler(string msg);
        LogMessageHandler myLogMessageHandler;

        private void RepositoryForm_Load(object sender, EventArgs e)
        {
            myLogMessageHandler = new LogMessageHandler(logMessageHandler);

            Directory.CreateDirectory(StoragePath);

            // Create a new instance of the Server
            server = new XdsMtomServer();
            
            // Set up server events
            server.ProvideAndRegisterRequestReceived += new RegisterDocumentSetHandler(server_ProvideAndRegisterRequestReceived);
            server.RetrieveRequestReceived += new RetrieveHandler(server_RetrieveRequestReceived);
            
            // Listen for incoming requests
            server.Listen(MyURI);

            xds = new XdsDomain();
            // Set up the Registry we are going to talk to
            xds.RegistryEndpoint = new WebServiceEndpoint(registryURI);
        }

        #region "Server Events"
        /// <summary>
        /// Server event to handle incoming Provide-and-Register request
        /// </summary>
        /// <param name="SubmissionSet">The SubmissionSet object which carries the documents (may also have documents within folders)</param>
        /// <returns>XdsRegistryResponse</returns>
        XdsRegistryResponse server_ProvideAndRegisterRequestReceived(XdsSubmissionSet SubmissionSet, XdsRequestInfo RequestInfo)
        {
            logMessageHandler("Provide and Register request received");

            // as a repository, we're only interested in the documents - nothing else!
            foreach (XdsDocument document in SubmissionSet.Documents)
            {
                // 1) Set Size and Hash
                document.SetSizeAndHash();

                // 2) Add our own repository OID
                document.RepositoryUniqueId = MyUID;

                // 3) Save Content for later (just use UUID as filename for now)
                File.WriteAllBytes(StoragePath + document.UniqueID, document.Data);

                // 4) Save the MimeType (in a real application - this would be in a database!)
                File.WriteAllText(StoragePath + document.UniqueID + ".mime", document.MimeType);
            }

            // then send whole lot off to the chosen registry (main data will NOT be sent - this is
            // automatic and internal!)  Then pass back the response

            // Pass back the Registry Response
            logMessageHandler("Sending Submissionset to Registry");
            return xds.RegisterDocumentSet(SubmissionSet);
        }

        /// <summary>
        /// Event to handle incoming document retrieve requests
        /// </summary>
        /// <param name="Request">Incoming Requests</param>
        /// <returns>XdsRetrieveResponse which carries matching documents</returns>
        XdsRetrieveResponse server_RetrieveRequestReceived(XdsRetrieveRequest Request, XdsRequestInfo RequestInfo)
        {
            logMessageHandler("Retreive document request received");

            XdsRetrieveResponse response = new XdsRetrieveResponse();
            response.Status = RegistryResponseStatus.Success; // gets changed if we have a failure

            foreach (XdsRetrieveItem item in Request.Requests)
            {
                // check that the stated repository OID really is us (this is an example of how to return an error)
                if (item.RepositoryUniqueID != MyUID)
                    return new XdsRetrieveResponse(XdsErrorCode.XDSUnknownRepositoryId, "", item.RepositoryUniqueID);

                // check the document exists
                // in a real world application, the matching could be done in database queries
                if (!File.Exists(StoragePath + item.DocumentUniqueID))
                    return new XdsRetrieveResponse(XdsErrorCode.XDSMissingDocument, "", item.DocumentUniqueID);

                // otherwise just pick up the document and mimetype
                XdsDocument document = new XdsDocument(StoragePath + item.DocumentUniqueID);
                document.MimeType = File.ReadAllText(StoragePath + item.DocumentUniqueID + ".mime");

                logMessageHandler("Returning requested document");
                response.Documents.Add(document);
            }
            return response;
        }
        #endregion

        void logMessageHandler(string msg)
        {
            if (InvokeRequired)
                Invoke(myLogMessageHandler, new object[] { msg });
            else
                logWindow.AppendText(msg + "\n");
        }
    }
}
