﻿using System;
using System.Windows.Forms;
using XdsObjects;
using XdsObjects.Enums;

namespace XDS_Document_Source__Simple_Version_
{
    public partial class DocConsumer : Form
    {
        public DocConsumer()
        {
            InitializeComponent();
            Setup();
        }

        #region "Variable"
        private XdsDomain xds;
        #endregion

        #region"Setup"
        private void Setup()
        {
            // Set up Logging
            XdsGlobal.LogToFile(@"c:\xds log files", LogLevel.FullXmlIn | LogLevel.FullXmlOut, 60);

            xds = new XdsDomain();
            // Where to send the query requests to
            xds.RegistryEndpoint = new WebServiceEndpoint("http://localhost:8090/MyRegistry");
            // Where to send the retrieve requests to
            xds.RepositoryAddresses.Add("1.19.6.24.109.42.1.5", new WebServiceEndpoint("http://localhost:8080/MyRepository"));
        }
        #endregion

        #region "UI Handler"

        private void FindDocBTN_Click(object sender, EventArgs e)
        {
            DocsTree.Nodes.Clear();
            XdsQueryResponse resp = Query();
            DisplayQueryResponse(resp, PIDText.Text);
        }

        /// <summary>
        /// Display the Query Result in the TreeView
        /// </summary>
        private void DisplayQueryResponse(XdsQueryResponse resp, string PatientID)
        {
            DocsTree.Nodes.Clear();

            TreeNode respNode = new TreeNode("Documents Found for Patient " + PatientID);

            // Display any errors returned by Registry
            TreeNode errorNode = respNode.Nodes.Add("Errors (" + resp.Errors.ErrorList.Count.ToString() + ")");
            foreach (XdsErrorItem err in resp.Errors.ErrorList)
            {
                errorNode.Nodes.Add("Severity : " + err.Severity + " codeContext: " + err.CodeContext);
            }

            // Display the Returned Status
            TreeNode statNode = respNode.Nodes.Add("Status :" + resp.Status.ToString());

            // Display the Returned List of Documents for this Patient
            TreeNode docNode = respNode.Nodes.Add("Returned Documents (" + resp.ReturnedDocuments.Count.ToString() + ")");
            foreach (XdsDocument doc in resp.ReturnedDocuments)
            {
                TreeNode node = docNode.Nodes.Add("MimeType: " + doc.MimeType + "   UID: " + doc.UniqueID);
                node.Tag = doc;
            }

            DocsTree.Nodes.Add(respNode);
            DocsTree.ExpandAll();
        }

        /// <summary>
        /// Double click on the tree item to send a retrieve request for the selected document
        /// </summary>
        private void DocsTree_DoubleClick(object sender, EventArgs e)
        {
            if (DocsTree.SelectedNode.Tag is XdsDocument)
                Retrieve((XdsDocument)DocsTree.SelectedNode.Tag);
        }

        #endregion

        #region "Query and Retrieve Operations"

        /// <summary>
        /// Query by Patient ID
        /// </summary>
        private XdsQueryResponse Query()
        {
            XdsQueryRequest q = new XdsQueryRequest();
            // Set the Patient ID
            q.PatientId = PIDText.Text;
            // Set the Query Return Type. LeafClass gives you everything where ObjectRef only gives you the reference of the object
            q.QueryReturnType = QueryReturnType.LeafClass;
            // Tpye of the Query operation
            q.QueryType = QueryType.FindDocuments;
            // The status of the documents to be returned
            q.DocumentEntryStatus = AvailabilityStatusCode.All;

            XdsQueryResponse resp = xds.RegistryStoredQuery(q);
            return resp;
        }

        /// <summary>
        /// Retrive the Document from NIST Public Repository and Display it
        /// </summary>        
        private void Retrieve(XdsDocument doc)
        {
            // Retrive the selected Document from Repository               
            XdsRetrieveRequest rq = new XdsRetrieveRequest();
            rq.Add(doc);

            // Display the downloaded Document into our Document Viewer
            XdsRetrieveResponse resp = xds.RetrieveDocumentSet(rq, doc.PatientInfo);
            if (resp.Documents[0].MimeType.ToUpper() == "APPLICATION/DICOM")
                MessageBox.Show("If the Document is a DICOM Image, you need a copy of DicomObjects to Display it. If the Document is a DICOM KOS object then we don't do anything in this sample.");
            else
                xdsDocumentViewer1.Display(resp.Documents[0]);
        }

        #endregion
    }
}
