﻿using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;
using DicomObjects;
using System.IO;
using System.Drawing.Printing;
using DicomObjects.Enums;
using DicomObjects.DicomUIDs;
using System.Windows.Media.Media3D;
using System.Drawing.Drawing2D;

namespace DicomObjects_V8_Sample_Viewer
{
    public partial class MainViewerForm : Form
    {
        // Local variables
        readonly OptionsBoxType OptionsBox = new OptionsBoxType();
        readonly RetrieveDialogType RetrieveBox = new RetrieveDialogType();
        readonly ImageInformationType AttributeForm = new ImageInformationType();
        readonly AboutBoxType AboutBox = new AboutBoxType();
        readonly LabelFormType LabelForm = new LabelFormType();
        DicomServer server;
        DicomImage SelectedImage, ActiveImage, MagnifierImage;
        bool LabelDrawing;
        PointF LastPosition, MouseDownPosition;
        ContentAlignment LabelEdge;
        int ActiveIndex;

        //  Labels to represent different Viewer functions
        DicomLabel CurrentLabel, PixelValueLabel, ValueLabel, CropLabel, FirstLineLabel;

        const byte BitFlag_2D = 1;      // 00000001
        const byte BitFlag_3D = 2;      // 00000010
        const byte BitFlag_BOTH = 3;    // 00000011

        private const string WheelFunction_Volume = "Move View Plane";
        private const string WheelFunction_NextImage = "Change Image";
        private const string WheelFunction_Cine = "Change Cine Frame";
        private const string LabelTag_Info = "INFO";
        private const string LabelTag_Ruler = "RULER";
        private const string LabelTag_Marker = "MARKER";
        private const string LabelTag_Value = "VALUE";
        private const string LabelText = "Frame: [Frame]\r\nCentre: [Level]\r\nWidth: [Width]\r\nZoom: [$MAG]";
        private const string LeftText = "LEFT";
        private const string TopText = "TOP";
        private const string BottomText = "BOTTOM";
        private const string RightText = "RIGHT";
        private const string InputBoxText = "Enter Label Text";
        private const string InputBoxTitle = "DicomObjects.NET Label Example";
        private const string AllFiles = "Image |*.bmp; *.png; *.dib; *.jpg; *.jpeg; *.tif; *.j2k; *.gif" +
                "|Video |*.avi; *.mpeg; *.mpg; *.mp4; *.mp2; *.mov; *.wmv" +
                "|Document |*.pdf; *.cda; *.xml; *.json; *.stl" +
                "|All files |*.*";
        private const string PdfFiles = "PDF |*.pdf";
        private const string VideoFiles = "Video |*.avi; *.mpeg; *.mpg; *.mp4; *.mp2; *.mov; *.wmv; *.gif";
        private const string ImageFiles = "Image |*.bmp; *.png; *.dib; *.jpg; *.jpeg; *.tif; *.j2k;";
        private const string OpenFileDialogTitleFor3DImages = "Open a single File containing a Multi-Frame DicomImage, or multiple single images using shift-key for multi-select";
        private const string AnonymiserMessageBoxText = "This is for demo purpose only to show how DicomObjects can be used to anonymise DICOM instances, and if required block off any burned-in patient demographics";
        private const string AnonymiserMessageBoxTitle = "Sample Image Anonymisation";
        private const string PixelValueLabel_NoImage = "No image under cursor.";
        private const string VerifyCECHOMessage = "Verify Returned Status --- ";
        private const string SendSelectedCSTOREMessage = "Returned Status = ";
        private const string SendAllCSTOREFailedMessage = "Send Failed - Returned Status = ";

        public enum MouseFunctionEnums
        {
            /*  These names should match the control on the Form for GroupBoxToEnum & EnumToGroupBox utility routines to work
             *  
             *  ODD values = 2D functions
             *  EVEN values = 3D functions
            */

            //  2D
            MagnifyingGlass = (1 << 2) | BitFlag_2D,
            PixelValues = (2 << 2) | BitFlag_2D,
            FreeRotation = (3 << 2) | BitFlag_2D,
            AddAnnotation = (4 << 2) | BitFlag_2D,
            EditAnnotation = (5 << 2) | BitFlag_2D,

            //  3D
            RotatePanZoom = (6 << 2) | BitFlag_3D,
            Crop = (7 << 2) | BitFlag_3D,

            //  BOTH
            Windowing = (8 << 2) | BitFlag_BOTH
        }

        public struct ViewerInteraction
        {
            public static readonly string AddLabel = "Add Label";
            public static readonly string AddMeasurement = "Add Measurement";
            public static readonly string Windowing = "Windowing";
            public static readonly string Scroll = "Scroll";
            public static readonly string Zoom = "Zoom";
            public static readonly string SelectLabel = "Select Label";
            public static readonly string Move = "Move";
            public static readonly string ReSize = "ReSize";
            public static readonly string Rotate_3D = "Rotate in 3D";
            public static readonly string Pan = "Pan";
            public static readonly string RectangleCrop = "Rectangle crop";
            public static readonly string LineCrop = "Line crop";
            public static readonly string Magnifier = "Magnifier";
            public static readonly string Pixel = "Show Pixel Values";
            public static readonly string Rotate_2D = "Free Rotation";
        }

        public enum WheelFunction
        {
            Volume,
            NextImage,
            Cine
        }

        static readonly Dictionary<WheelFunction, string> WheelNames = new Dictionary<WheelFunction, string>()
        {
            { WheelFunction.Volume, WheelFunction_Volume },
            { WheelFunction.NextImage, WheelFunction_NextImage },
            { WheelFunction.Cine, WheelFunction_Cine }
        };

        static readonly Dictionary<MouseFunctionEnums, string[]> MouseFunctionNames = new Dictionary<MouseFunctionEnums, string[]>()
        {
            { MouseFunctionEnums.AddAnnotation,
                new string[] { ViewerInteraction.AddLabel, ViewerInteraction.AddLabel, ViewerInteraction.AddMeasurement } },
            { MouseFunctionEnums.Windowing,
                new string[] { ViewerInteraction.Windowing, ViewerInteraction.Scroll, ViewerInteraction.Zoom } },
            { MouseFunctionEnums.EditAnnotation,
                new string[] { ViewerInteraction.SelectLabel, ViewerInteraction.Move, ViewerInteraction.ReSize } },
            { MouseFunctionEnums.RotatePanZoom,
                new string[] { ViewerInteraction.Rotate_3D, ViewerInteraction.Pan, ViewerInteraction.Zoom } },
            { MouseFunctionEnums.Crop,
                new string[] { ViewerInteraction.RectangleCrop, ViewerInteraction.Pan, ViewerInteraction.LineCrop } },
            { MouseFunctionEnums.MagnifyingGlass,
                new string[] { ViewerInteraction.Magnifier, ViewerInteraction.Magnifier, ViewerInteraction.Magnifier } },
            { MouseFunctionEnums.PixelValues,
                new string[] { ViewerInteraction.Pixel, ViewerInteraction.Pixel, ViewerInteraction.Pixel } },
            { MouseFunctionEnums.FreeRotation,
                new string[] { ViewerInteraction.Rotate_2D, ViewerInteraction.Rotate_2D, ViewerInteraction.Rotate_2D } },
        };

        [STAThread]
        static void Main()
        {
            Application.Run(new MainViewerForm());
        }

        public MainViewerForm()
        {
            InitializeComponent();
        }

        private void MainViewerForm_Load(object sender, EventArgs e)
        {
            if (!Properties.Settings.Default.DialogShown)
            {
                AboutBox.ShowDialog();
                Properties.Settings.Default.DialogShown = true;
                Properties.Settings.Default.Save();
            }

            // set up simplistic SCP behaviour - accept and display images
            server = new DicomServer();
            server.Listen(int.Parse(OptionsBox.Port.Text));
            server.InstanceReceived += (o, ee) => { Viewer.Images.Add(new DicomImage(ee.Instance)); ee.Status = 0; };
            server.VerifyReceived += (o, ee) => ee.Status = 0;
            OptionsBox.ServerOpt = server;

            //Initially hide cinemode toolbar
            EnableCinemodeToolbar();

            //Show the viewermode in the toolbar button
            SetViewerMode(ViewerMode.DirectX);

            //  odd-  behaviour - this cannot be set in the designer, and must be set here.
            Viewer.MouseWheel += Viewer_MouseWheel;

            //Show 2dControls by default
            UpdateControls();
        }

        private WheelFunction GetWheelFunction()
        {
            if (SelectedImage is IProjection)
                return WheelFunction.Volume;
            if (SelectedImage?.FrameCount > 1)
                return WheelFunction.Cine;
            return WheelFunction.NextImage;
        }

        private void Viewer_MouseWheel(object sender, MouseEventArgs e)
        {
            switch (GetWheelFunction())
            {
                case WheelFunction.Volume:
                    (SelectedImage as IProjection).TranslateNormal(Math.Sign(e.Delta) * 2); // 2mm increments
                    break;

                case WheelFunction.NextImage:
                    // increment/decrement within range
                    Viewer.CurrentIndex = Math.Max(0, Math.Min(Viewer.Images.Count - 1, Viewer.CurrentIndex + Math.Sign(e.Delta)));
                    break;

                case WheelFunction.Cine:
                    GetSelectedImage().Frame = Math.Max(1, Math.Min(GetSelectedImage().FrameCount, GetSelectedImage().Frame + Math.Sign(e.Delta)));
                    break;
            }
        }

        private DicomImage GetSelectedImage()
        {
            if (SelectedImage == null)
                SetSelectedImage(Viewer.CurrentImage);

            return SelectedImage;
        }

        private MouseFunctionEnums MouseFunction
        {
            get { return Utilities.GroupBoxToEnum<MouseFunctionEnums>(grpMouseFunctionGroup); }
        }

        private void UpdateImages(Action<DicomImage> action)
        {
            if (OptionsBox.WindowAll.Checked)
                foreach (DicomImage image in Viewer.Images)
                    action(image);
            else
                UpdateImage(action);
        }

        private void UpdateImage(Action<DicomImage> action)
        {
            var image = GetSelectedImage();
            if (image != null)
                action(image);
        }

        private void SetSmoothingAll(bool Smooth)
        {
            // this is a simple version for the demo - other smoothing modes exist
            if (Smooth)
            {
                UpdateImages(i => i.MagnificationMode = FilterMode.BSpline);
                UpdateImages(i => i.MinificationMode = FilterMode.MovingAverage);
            }
            else
            {
                UpdateImages(i => i.MagnificationMode = FilterMode.Replicate);
                UpdateImages(i => i.MinificationMode = FilterMode.Replicate);
            }
        }

        private void UpdateInformation()
        {
            foreach (DicomImage image in Viewer.Images)
            {
                // remove existing
                image.Labels.Remove(image.Labels.Where(label => (string)label.Tag == LabelTag_Info).ToList());
                UpdateDicomLabels(image);
                
            }
        }

        private void UpdateDicomLabels(DicomImage image)
        {
            if (ShowPatientAndExamDetails.Checked)
            {
                DicomLabel label = DataLabel(image);
                label.Left = 0;
                label.Top = 0;
                label.Text = $"{image.Name}\r\n{(image[Keyword.PatientBirthDate].ExistsWithValue ? image[Keyword.PatientBirthDate].Value.ToString() : "")}";
                label.StringFormat = new StringFormat() { Alignment = StringAlignment.Near, LineAlignment = StringAlignment.Near }; // top left

                label = DataLabel(image);
                label.Left = 0.5F;
                label.Top = 0;
                label.Text = $"{image.StudyDescription}\r\n{(image[Keyword.StudyDate].ExistsWithValue ? image[Keyword.PatientBirthDate].Value : "")}\r\n{(image[Keyword.StudyTime].ExistsWithValue ? image[Keyword.StudyTime].Value : "")}";
                label.StringFormat = new StringFormat() { Alignment = StringAlignment.Far, LineAlignment = StringAlignment.Near }; // top right

                label = DataLabel(image);
                label.Left = 0;
                label.Top = 0.5F;
                var text = $@"{(image[Keyword.Modality].ExistsWithValue ? image[Keyword.Modality].Value.ToString() : "")}
                                {(image[Keyword.Manufacturer].ExistsWithValue ? "\r\n" + image[Keyword.Manufacturer].Value.ToString() : "")}
                                {(image[Keyword.InstitutionName].ExistsWithValue ? "\r\n" + image[Keyword.InstitutionName].Value.ToString() : "") }";
                label.Text = text;
                label.StringFormat = new StringFormat() { Alignment = StringAlignment.Near, LineAlignment = StringAlignment.Far }; // bottom left

                // this one does not need to be manually updated - it is updated dynamically on every refresh
                if (image.DataSet[Keyword.PixelData].ExistsWithValue)
                {
                    label = DataLabel(image);
                    label.Left = 0.5F;
                    label.Top = 0.5F;
                    label.Text = LabelText;
                    label.LabelType = LabelType.Formatted;
                    label.StringFormat = new StringFormat() { Alignment = StringAlignment.Far, LineAlignment = StringAlignment.Far }; // bottom right
                }
            }
        }

        private DicomLabel DataLabel(DicomImage Image)
        {
            DicomLabel label = new DicomLabel()
            {
                LabelType = LabelType.Text,
                ScaleMode = ScaleMode.Cell,
                ScaleFontSize = true,
                Area = new RectangleF(0, 0, 0.5F, 0.5F),
                TextBrush = Brushes.LightGoldenrodYellow,
                Margin = 8,
                Tag = LabelTag_Info
            };

            Image.Labels.Add(label);
            return label;
        }

        private void UpdateRulers()
        {
            foreach (DicomImage Image in Viewer.Images)
            {
                // remove existing
                Image.Labels.Remove(Image.Labels.Where(label => (string)label.Tag == LabelTag_Ruler).ToList());

                if (ShowRuler.Checked)
                {
                    DicomLabel label = new DicomLabel()
                    {
                        LabelType = LabelType.Ruler,
                        ScaleMode = ScaleMode.Cell,
                        Font = new Font(FontFamily.GenericSansSerif, 20),
                        Area = new RectangleF(0.01F, 0.2F, 0.03F, 0.6F),
                        Pen = new Pen(Color.Aqua, 0),
                        Tag = LabelTag_Ruler
                    };
                    Image.Labels.Add(label);
                }
            }
        }

        private void UpdateMarkers()
        {
            foreach (DicomImage Image in Viewer.Images)
            {
                // remove existing
                Image.Labels.Remove(Image.Labels.Where(item => (string)item.Tag == LabelTag_Marker).ToList());

                if (AnatomicMarkers.Checked)
                {
                    DicomLabel label = MarkerLabel(Image);
                    label.Left = 0.05F;
                    label.Top = 0.25F;
                    label.Text = LeftText;
                    label.StringFormat = new StringFormat() { Alignment = StringAlignment.Near, LineAlignment = StringAlignment.Center }; // top left

                    label = MarkerLabel(Image);
                    label.Left = 0.25F;
                    label.Top = 0.05F;
                    label.Text = TopText;
                    label.StringFormat = new StringFormat() { Alignment = StringAlignment.Center, LineAlignment = StringAlignment.Near }; // top left

                    label = MarkerLabel(Image);
                    label.Left = 0.5F;
                    label.Top = 0.25F;
                    label.Text = RightText;
                    label.StringFormat = new StringFormat() { Alignment = StringAlignment.Far, LineAlignment = StringAlignment.Center }; // top left

                    label = MarkerLabel(Image);
                    label.Left = 0.25F;
                    label.Top = 0.5F;
                    label.Text = BottomText;
                    label.StringFormat = new StringFormat() { Alignment = StringAlignment.Center, LineAlignment = StringAlignment.Far }; // top left
                }
            }
        }

        private DicomLabel MarkerLabel(DicomImage Image)
        {
            DicomLabel label = new DicomLabel()
            {
                LabelType = LabelType.Special,
                ScaleMode = ScaleMode.Cell,
                ScaleFontSize = true,
                Font = new Font(FontFamily.GenericSansSerif, 20),
                Area = new RectangleF(0, 0, 0.45F, 0.45F),
                TextBrush = Brushes.Red,
                Tag = LabelTag_Marker
            };
            Image.Labels.Add(label);
            return label;
        }

        private void EndMagnifier()
        {
            if (MagnifierImage != null)
            {
                Viewer.Images.Remove(MagnifierImage);
                MagnifierImage = null;
            }
        }

        private void SetMagnifier(Point p)
        {
            if (ActiveImage != null)
            {
                if (MagnifierImage == null)
                {
                    MagnifierImage = ActiveImage.Clone(true) as DicomImage;
                    MagnifierImage.StretchMode = StretchModes.NoStretch;
                    Viewer.Images.Add(MagnifierImage);
                    MagnifierImage.Zoom = DicomGlobal.Zoom(ActiveImage.Matrix(Viewer)) * 2;
                    MagnifierImage.Labels.Clear();
                }

                float MagnifierSize = 0.2F;
                RectangleF Rect = new RectangleF(p.X / (float)Viewer.ClientSize.Width, p.Y / (float)Viewer.ClientSize.Height, 0, 0);
                Rect.Inflate(MagnifierSize / 2, MagnifierSize / 2);
                MagnifierImage.Area = Rect;

                // Work out how far to scroll the magnified image by doing screen ==> Image==> Screen image coordinates
                // not simply done by scaling of offset, as anisometropic pixels can cause problems
                var ImageCoords = ActiveImage.ScreenToImage(p, Viewer).Value;
                var MagCoords = MagnifierImage.ImageToScreen(ImageCoords, Viewer).Value;

                MagnifierImage.Scroll = ((PointF)p) + new SizeF((MagnifierImage.Scroll - new SizeF(MagCoords)));
            }
        }

        private void AboutThisExampleToolStripMenuItem_Click(object sender, EventArgs e)
        {
            AboutBox.ShowDialog();
        }

        private void StretchToFit_CheckedChanged(object sender, EventArgs e)
        {
            UpdateImage(image =>
            {
                if (StretchToFit.Checked)
                    image.StretchMode = StretchModes.StretchCentred;
                else
                {
                    //reset scroll & zoom to match previous stretched version
                    Matrix matrix = image.Matrix(Viewer);
                    image.Zoom = DicomGlobal.Zoom(matrix);
                    image.Scroll = new PointF(matrix.OffsetX, matrix.OffsetY);
                    image.StretchMode = StretchModes.NoStretch;
                }
            });
        }

        private void CImageIndex_MouseUp(object sender, MouseEventArgs e)
        {
            if (Viewer.Images.Count > 0)
            {
                Viewer.CurrentIndex = (int)CImageIndex.Value;
            }
        }

        private void Rows_MouseUp(object sender, MouseEventArgs e)
        {
            Viewer.MultiRows = (short)Rows.Value;
        }

        private void Columns_MouseUp(object sender, MouseEventArgs e)
        {
            Viewer.MultiColumns = (short)Columns.Value;
        }

        private void AnatomicMarkers_CheckedChanged(object sender, EventArgs e)
        {
            UpdateMarkers();
        }

        private void Viewer_DataChanged(object Sender)
        {
            Rows.Value = Viewer.MultiRows;
            Columns.Value = Viewer.MultiColumns;
            CImageIndex.Maximum = Viewer.Images.Count - 1;

            if (Viewer.Images.Count > 0)
            {
                CImageIndex.Value = Viewer.CurrentIndex;
                bool PreState = GetSelectedImage()?.PresentationState != null;
                StretchToFit.Visible = !PreState;
                StretchToFit.Checked = GetSelectedImage()?.StretchMode != StretchModes.NoStretch;
            }
            else
            {
                StretchToFit.Visible = false;
            }

            UpdateMarkers();
            UpdateInformation();
            UpdateRulers();
            EnableCinemodeToolbar();
        }

        private void Viewer_DisplayChanged(object Sender, Region Area)
        {
            MouseWheelFunction.Text = WheelNames[GetWheelFunction()];
        }

        private void Viewer_MouseDown(object sender, MouseEventArgs e)
        {
            Viewer.Focus();

            // update active irrespective of mouse function
            UpdateImage(i => i.BorderPen = Pens.Transparent); // clear previous

            ActiveIndex = Viewer.ImageIndex(e.Location);
            if (ActiveIndex >= 0 && ActiveIndex < Viewer.Images.Count)
            {
                ActiveImage = Viewer.Images[ActiveIndex];
                SetSelectedImage(ActiveImage); // this one is NOT cleared in mouse up - used in GetSelectedImage
            }
            else
            {
                ActiveImage = null;
                //If this is a new label action then we need to process it and not return here
                if (LabelDrawing) return;
            }

            MouseDownPosition = e.Location;

            switch (MouseFunction)
            {
                case MouseFunctionEnums.Windowing:
                    if (DisableSmoothing.Checked)
                        SetSmoothingAll(false);
                    break;

                case MouseFunctionEnums.AddAnnotation:
                    ViewerMouseDown_AddAnnotation(e);
                    break;

                case MouseFunctionEnums.EditAnnotation:
                    ViewerMouseDown_EditAnnotation(e);
                    break;

                case MouseFunctionEnums.MagnifyingGlass:
                    SetMagnifier(e.Location);
                    break;

                case MouseFunctionEnums.PixelValues:
                    SetPixelLabel(e.Location);
                    break;

                case MouseFunctionEnums.Crop:
                    ViewerMouseDown_Crop(e);
                    break;

            } // End switch
            LastPosition = e.Location;
        }


        #region ViewerMouseDown functions
        private void ViewerMouseDown_AddAnnotation(MouseEventArgs e)
        {
            if (CurrentLabel != null)
                CurrentLabel.SelectMode = SelectMode.None;

            CurrentLabel = LabelForm.NewLabel(e.Location);
            LabelDrawing = true;
            Viewer.Labels.Add(CurrentLabel);

            if (e.Button == MouseButtons.Right || LabelForm.IsAngle()) //Measure
            {
                if (FirstLineLabel == null && LabelForm.IsAngle())
                {
                    FirstLineLabel = CurrentLabel;
                }
                else
                {
                    ValueLabel = new DicomLabel()
                    {
                        ScaleMode = ScaleMode.Output,
                        ScaleFontSize = false,
                        TextBrush = Brushes.YellowGreen,
                        Font = LabelForm.fontDialog.Font,
                        Tag = LabelTag_Value,
                    };

                    if (!LabelForm.Transparent.Checked)
                        ValueLabel.Brush = new SolidBrush(LabelForm.BackColour.BackColor);

                    Viewer.Labels.Add(ValueLabel);
                }
            }
        }

        private void ViewerMouseDown_EditAnnotation(MouseEventArgs e)
        {
            switch (e.Button)
            {

                case MouseButtons.Left://select label
                    var labelList = Viewer.LabelHits(e.Location, true, true).Where(i => i.Tag == null).ToList(); // exclude markers etc.

                    if (CurrentLabel != null)
                    {
                        CurrentLabel.SelectMode = SelectMode.None;
                        CurrentLabel = null;
                    }

                    if (labelList.Count > 0)
                    {
                        //The line below can be modified to refer to any label within the collection
                        //This example selects the first label within the collection
                        CurrentLabel = labelList[0];
                        CurrentLabel.SelectPen = Pens.YellowGreen;
                        CurrentLabel.SelectMode = SelectMode.Box;
                    }
                    break;
                case MouseButtons.Right://Right Button to Resize Label
                    ResizeLabel(e);
                    break;
                case MouseButtons.Middle:// Middle Button to Move the Label
                    LabelEdge = ContentAlignment.MiddleCenter;
                    break;
            }

        }

        private void ViewerMouseDown_Crop(MouseEventArgs eventArgs)
        {
            //  Initialize Crop Label with a starting point
            CropLabel = LabelForm.NewLabel(eventArgs.Location);

            //  Add label to the 
            Viewer.Labels.Add(CropLabel);

            switch (eventArgs.Button)
            {
                //  Rectangle crop
                case MouseButtons.Left:
                    CropLabel.LabelType = LabelType.Rectangle;
                    break;

                //  Line crop
                case MouseButtons.Right:
                    CropLabel.LabelType = LabelType.Line;
                    break;
            }
        }

        #endregion


        private void ResizeLabel(MouseEventArgs e)
        {
            if (CurrentLabel != null)
            {
                PointF ImageRelativePosition = e.Location;
                if (CurrentLabel.ScaleMode == ScaleMode.Image)
                    ImageRelativePosition = GetSelectedImage().ScreenToImage(e.Location, Viewer).Value;

                float x = (ImageRelativePosition.X - CurrentLabel.BoundingBox.Left) / CurrentLabel.BoundingBox.Width;
                float y = (ImageRelativePosition.Y - CurrentLabel.BoundingBox.Top) / CurrentLabel.BoundingBox.Height;

                //int H = (x < 0.3F) ? 0 : ((x < 0.7F) ? 1 : 2); // 0 = left, 1 = middle, 2 = right
                //int V = (y < 0.3F) ? 0 : ((y < 0.7F) ? 4 : 8); // 0 = top, 4 = middle, 8 = Bottom

                int H = 0;
                if (x < 0.3F) H = 0;
                else if (x < 0.7F) H = 1;
                else H = 2;

                int V = 0;
                if (y < 0.3F) V = 0;
                else if (y < 0.7F) V = 4;
                else V = 8;

                LabelEdge = (ContentAlignment)(1 << (H + V)); // odd, but that's how ContentAlignment is defined!
            }
        }

        private void Viewer_MouseMove(object sender, MouseEventArgs e)
        {
            if ((ActiveIndex < 0 || !Viewer.CellArea(ActiveIndex).Contains(e.Location) && e.Button != MouseButtons.None) 
                    && MouseFunction != MouseFunctionEnums.EditAnnotation && MouseFunction != MouseFunctionEnums.AddAnnotation)// this version allows for overlapping images sauch as the magnifier
            {
                Viewer_MouseUp(sender, e);
                Viewer_MouseDown(sender, e);
            }

            //This is to ensure we don't select a new active image when we are simply moving the mouse
            // over an image without clicking/dragging or anything else.
            if (!((MouseFunction == MouseFunctionEnums.AddAnnotation || MouseFunction == MouseFunctionEnums.EditAnnotation) && LabelForm.ScaleMode == ScaleMode.Output)
                && (e.Button == MouseButtons.None || ActiveImage == null))
                return;

            Point Position = e.Location;
            SizeF Offset = new SizeF(e.X - LastPosition.X, e.Y - LastPosition.Y);

            if (e.Button > 0)
            {
                switch (MouseFunction)
                {
                    case MouseFunctionEnums.Windowing:
                        ViewerMouseMove_Windowing(e, Offset);
                        break;

                    case MouseFunctionEnums.AddAnnotation:
                        ViewerMouseMove_AddAnnotation(e);
                        break;

                    case MouseFunctionEnums.EditAnnotation:
                        ViewerMouseMove_EditAnnotation(e, Offset);
                        break;

                    case MouseFunctionEnums.MagnifyingGlass:
                        // if user has strayed onto a different image, then simulate mouse up and then down again - but only for this mouse function.
                        if (ActiveImage != null)
                        {
                            SetMagnifier(Position);
                        }
                        break;

                    case MouseFunctionEnums.PixelValues:
                        SetPixelLabel(Position);
                        break;

                    case MouseFunctionEnums.FreeRotation:
                        // just use horizontal movement to rotate
                        ActiveImage.Angle += Offset.Width;
                        break;

                    case MouseFunctionEnums.RotatePanZoom:
                        ViewerMouseMove_RotatePanZoom(e, Offset);
                        break;

                    case MouseFunctionEnums.Crop:
                        ViwerMouseMove_Crop(e, Offset);
                        break;
                }
            }
            LastPosition = Position;
        }

        #region ViwerMouseMove functions
        private void ViewerMouseMove_Windowing(MouseEventArgs eventArgs, SizeF offset)
        {
            switch (eventArgs.Button)
            {
                case MouseButtons.Left: // windowing
                    ActiveImage.Width += offset.Width;
                    ActiveImage.Level += offset.Height;
                    break;

                case MouseButtons.Middle: // scrolling        
                    ActiveImage.Scroll += offset;
                    break;

                case MouseButtons.Right: // zooming

                    // 1) Find the IMAGE coordinates of the reference point 
                    var ImageReferencePoint = ActiveImage.ScreenToImage(MouseDownPosition, Viewer);

                    if (ImageReferencePoint.HasValue)
                    {
                        // 2) Change Image Zoom
                        // This is an arbitrary calculation to give a reasonable exponential zoom range
                        ActiveImage.Zoom *= (float)Math.Pow(1.01, offset.Width);

                        // 5) Calculate the offset to modify Image's Scroll property to leave Reference point unchanged on the screen
                        var ScreenLocation = ActiveImage.ImageToScreen(ImageReferencePoint.Value, Viewer);
                        SizeF Adjustment = new SizeF(MouseDownPosition) - new SizeF(ScreenLocation.Value);
                        ActiveImage.Scroll += Adjustment;
                    }
                    break;
            }
        }

        private void ViewerMouseMove_AddAnnotation(MouseEventArgs eventArgs)
        {
            if (LabelDrawing)
            {
                CurrentLabel.Area = RectangleF.FromLTRB(CurrentLabel.Area.Left, CurrentLabel.Area.Top, eventArgs.Location.X, eventArgs.Location.Y);
                if (CurrentLabel.LabelType == LabelType.Polygon || CurrentLabel.LabelType == LabelType.PolyLine)
                    CurrentLabel.AddPoint(eventArgs.Location);

                if (ValueLabel != null) //Measurement in progress
                {
                    // handle angle differently and leave in viewer corrdinates (as these are already compensated for nonsquare pixels)
                    if (FirstLineLabel != null && FirstLineLabel != CurrentLabel)
                    {
                        ValueLabel.Text = $"{Math.Abs(Math.Atan2(FirstLineLabel.Height, FirstLineLabel.Width) - Math.Atan2(CurrentLabel.Height, CurrentLabel.Width)) * 180 / Math.PI:#.#}°";
                    }
                    else
                    {
                        // temporarily convert to Image coordinates for ROI
                        CurrentLabel.Rescale(Viewer, ActiveIndex, false, ScaleMode.Image);
                        switch (CurrentLabel.LabelType)
                        {
                            case LabelType.Line:
                            case LabelType.PolyLine:
                                ValueLabel.Text = $"{CurrentLabel.ROILength(ActiveImage):####.#} { CurrentLabel.ROIDistanceUnits(ActiveImage)}";
                                break;
                            case LabelType.Polygon:
                            case LabelType.Rectangle:
                            case LabelType.Ellipse:
                                ValueLabel.Text = $"Area = {CurrentLabel.ROIArea(Viewer, Viewer.CurrentIndex):#.#} {CurrentLabel.ROIDistanceUnits(ActiveImage)}²"
                                                + $"\r\nMean value = {CurrentLabel.ROIMean(ActiveImage, true):#.#}";
                                break;
                            default:
                                ValueLabel.Text = string.Empty;
                                break;
                        }
                        CurrentLabel.Rescale(Viewer, ActiveIndex, false, ScaleMode.Output);
                    }
                    ValueLabel.Area = new RectangleF(eventArgs.Location.X, eventArgs.Location.Y - 20, 100, 100);
                }
            }
        }

        private void ViewerMouseMove_EditAnnotation(MouseEventArgs eventArgs, SizeF offset)
        {
            if (CurrentLabel == null)
                return;

            if (eventArgs.Button == MouseButtons.Right || eventArgs.Button == MouseButtons.Middle)
            {
                int tempIndex = (ActiveIndex > 0 && ActiveIndex < Viewer.Images.Count) ? ActiveIndex : 0;
                CurrentLabel.Adjust(LabelEdge, offset, Viewer, tempIndex);
            }
        }

        private void ViewerMouseMove_RotatePanZoom(MouseEventArgs eventArgs, SizeF offset)
        {
            IProjection Selected3DImage = ActiveImage as IProjection;

            switch (eventArgs.Button)
            {
                // Rotate about vertical and horizontal axes
                case MouseButtons.Left:
                    Selected3DImage?.Rotate(offset.Width / 2.0F, new Vector3D(0, -1, 0));
                    Selected3DImage?.Rotate(offset.Height / 2.0F, new Vector3D(1, 0, 0));
                    break;

                // Pan
                case MouseButtons.Middle:
                    Selected3DImage?.TranslatePixels(offset, ActiveImage.DisplaySize(Viewer));
                    break;

                // up/down = Zoom
                // Left/right = Rotate about Z axis
                case MouseButtons.Right:
                    Selected3DImage?.Rotate(offset.Width / 2.0F, new Vector3D(0, 0, 1));
                    Selected3DImage?.Scale(1 + (offset.Height / 200.0F));
                    break;
            }
        }

        private void ViwerMouseMove_Crop(MouseEventArgs eventArgs, SizeF offset)
        {
            IProjection Selected3DImage = ActiveImage as IProjection;
            switch (eventArgs.Button)
            {
                //  Move Rectangle/Line Crop label
                case MouseButtons.Left:
                case MouseButtons.Right:
                    if (CropLabel != null)
                        CropLabel.Area = RectangleF.FromLTRB(MouseDownPosition.X, MouseDownPosition.Y, eventArgs.X, eventArgs.Y);
                    break;

                //  Rotate about vertical and horizontal axes
                case MouseButtons.Middle:
                    Selected3DImage?.Rotate(offset.Width / 2.0F, new Vector3D(0, -1, 0));
                    Selected3DImage?.Rotate(offset.Height / 2.0F, new Vector3D(1, 0, 0));
                    break;
            }
        }
        #endregion



        private void Viewer_MouseUp(object sender, MouseEventArgs e)
        {
            switch (MouseFunction)
            {
                case MouseFunctionEnums.Windowing:
                    SetSmoothingAll(SmoothImages.Checked);
                    break;

                case MouseFunctionEnums.MagnifyingGlass:
                    EndMagnifier();
                    break;

                case MouseFunctionEnums.PixelValues:
                    Viewer.Labels.Remove(PixelValueLabel);
                    PixelValueLabel = null;
                    break;

                case MouseFunctionEnums.AddAnnotation:
                    ViewerMouseUp_AddAnnotation();
                    break;

                case MouseFunctionEnums.Crop:
                    var Selected3DImage = ActiveImage as DicomImage3D;

                    //  Only valid for MIP and VR
                    Selected3DImage?.Clip(CropLabel, Selected3DImage.CellSize(Viewer));
                    Viewer.Labels.Clear();
                    CropLabel = null;
                    break;
            }
            ActiveImage = null;
        }
        
        private void ViewerMouseUp_AddAnnotation()
        {
            // handle angle separately as it uses TWO lines
            if (LabelForm.IsAngle())
            {
                if (FirstLineLabel == CurrentLabel) // this is the first - leave in viewer units
                {
                    CurrentLabel = null;
                    return;
                }
                else
                {
                    MoveLabelViewerToImage(FirstLineLabel);
                    FirstLineLabel = null;
                }
            }

            LabelDrawing = false;

            if (LabelForm.ScaleMode == ScaleMode.Output) // attach currentlabel to viewer
            {
                Viewer.Labels.Add(CurrentLabel);
            }
            else // attach currentlabel to target image
            {
                if (ActiveImage != null)
                {
                    if (ValueLabel != null) //Measure
                    {
                        MoveLabelViewerToImage(ValueLabel);
                        ValueLabel = null;
                    }

                    MoveLabelViewerToImage(CurrentLabel);
                }
            }

            if (CurrentLabel != null && CurrentLabel.LabelType == LabelType.Text)
            {
                CurrentLabel.Font = LabelForm.fontDialog.Font;
                CurrentLabel.Text = Microsoft.VisualBasic.Interaction.InputBox(InputBoxText, InputBoxTitle, "", 500, 500);
            }
            CurrentLabel = null;
        }

        private void MoveLabelViewerToImage(DicomLabel Label)
        {
            Viewer.Labels.Remove(Label);
            ActiveImage.Labels.Add(Label);
                Label.Rescale(Viewer, ActiveIndex, true, LabelForm.ScaleMode);
            }

        private void SmoothImages_CheckedChanged(object sender, EventArgs e)
        {
            SetSmoothingAll(SmoothImages.Checked);
        }

        private void Load2DToolStripMenuItem_Click(object sender, EventArgs e)
        {
            OpenFileDialog openFileDialog = new OpenFileDialog
            {
                Multiselect = true
            };
            var samplePath = Path.Combine(SampleExtensions.GetSampleImageFolderPath(), SampleExtensions.IMAGE_FOLDER_2D);

            if (Directory.Exists(samplePath))
                openFileDialog.InitialDirectory = samplePath;

            if (openFileDialog.ShowDialog() == DialogResult.OK)
            {
                Viewer.Images.Read(openFileDialog.FileNames);
                SetSmoothingAll(SmoothImages.Checked);
                EnableCinemodeToolbar();
                SetSelectedImage(Viewer.CurrentImage);
                Viewer.AdjustMultiRowsColumns();
            }
        }

        private void SetSelectedImage(DicomImage image)
        {
            SelectedImage = image;
            if (SelectedImage != null)
            {
                UpdateControls();
                StretchToFit.Checked = SelectedImage.StretchMode != StretchModes.NoStretch;
                UpdateImage(i => i.BorderPen = Pens.Red);
            }
        }

        void UpdateControls()
        {
            bool ImageIs3D = SelectedImage is DicomImage3D;
            TabHolder2D3D.SelectedIndex = ImageIs3D ? 1 : 0;

            foreach (MouseFunctionEnums function in Enum.GetValues(typeof(MouseFunctionEnums)))
            {
                Control c = grpMouseFunctionGroup.Controls[function.ToString()];
                if (c != null)
                {
                    //  Checking for ODD/EVEN (2D/3D functions) enum values
                    if (((int)function & BitFlag_BOTH) == BitFlag_BOTH)  // BitFlag_BOTH is set
                    {
                        //  One or the other will enable this control, because it supports both 2D & 3D functions!
                        c.Enabled = true;
                    }
                    else if (((int)function & BitFlag_2D) == BitFlag_2D) // BitFlag_2D is set
                    {
                        c.Enabled = !ImageIs3D;
                    }
                    else // BitFlag_3D is set
                    {
                        c.Enabled = ImageIs3D;
                    }
                }
            }
        }

        private void WriteToolStripMenuItem_Click(object sender, EventArgs e)
        {
            if (GetSelectedImage() == null) return;
            SaveFileDialog saveFileDialog = new SaveFileDialog();
            if (saveFileDialog.ShowDialog() == DialogResult.OK)
            {
                GetSelectedImage()?.Write(saveFileDialog.FileName, TransferSyntaxes.ImplicitVRLittleEndian);
            }
        }

        private void ApplyPresentationStateToolStripMenuItem_Click(object sender, EventArgs e)
        {
            var openFileDialog = new OpenFileDialog();
            if (openFileDialog.ShowDialog() == DialogResult.OK)
            {
                UpdateImage(i => i.PresentationState = new DicomDataSet(openFileDialog.FileName));
            }
        }

        private void MainViewerForm_FormClosing(object sender, FormClosingEventArgs e)
        {
            server.UnlistenAll();
        }

        private void PrintDICOMToolStripMenuItem_Click(object sender, EventArgs e)
        {
            if (Viewer.Images.Count <= 0) return;
            DicomPrint printer = new DicomPrint()
            {
                Colour = OptionsBox.PrintInColour.Checked,
                Format = OptionsBox.PrintFormat.Text,
                Orientation = OptionsBox.PrintOrientation.Text,
                FilmSize = OptionsBox.FilmSize.Text
            };

            printer.Open(OptionsBox.PrintNode.Text, Convert.ToInt32(OptionsBox.PrintPort.Text), OptionsBox.PrintAE.Text, OptionsBox.PrintCallingAE.Text);
            printer.FilmBox.Add(Keyword.MagnificationType, OptionsBox.Magnification.Text);

            //  Sending all images loaded on Viewer for printing
            foreach (DicomImage img in Viewer.Images)
                printer.PrintImage(img, true, true);
            printer.Close();
        }

        private void PrintWindowsToolStripMenuItem_Click(object sender, EventArgs e)
        {
            DicomImage image = GetSelectedImage();
            if (image == null) return;
            PrintDocument printDoc = new PrintDocument();
            printDoc.PrintPage += (o, ee) =>
            {
                Image printImage = image.Bitmap();
                ee.Graphics.DrawImage(printImage, 0, 0);
            };
            printDoc.Print();
        }

        private void ImportFromNonDICOMFileToolStripMenuItem_Click(object sender, EventArgs e)
        {
            OpenFileDialog openFileDialog = new OpenFileDialog
            {
                Filter = AllFiles
            };

            if (openFileDialog.ShowDialog() == DialogResult.OK)
            {
                DicomImage image = new DicomImage();
                image.Import(openFileDialog.FileName);
                Viewer.Images.Add(image);
            }
        }

        private void ExportToNonDICOMFileToolStripMenuItem_Click(object sender, EventArgs e)
        {
            if (GetSelectedImage() == null) return;
            SaveFileDialog saveFileDialog1 = new SaveFileDialog();

            if (GetSelectedImage()?.SOPClass == SOPClasses.EncapsulatedPDF)
                saveFileDialog1.Filter = PdfFiles;
            else if (GetSelectedImage()?.FrameCount > 1)
                saveFileDialog1.Filter = VideoFiles;
            else
                saveFileDialog1.Filter = ImageFiles;

            if (saveFileDialog1.ShowDialog() == DialogResult.OK)
            {
                // this is basic - lots of options exist
                GetSelectedImage()?.Export(saveFileDialog1.FileName);
            }
        }

        private void ExitToolStripMenuItem_Click(object sender, EventArgs e)
        {
            Close();
        }

        private void DeleteSelectedImageToolStripMenuItem_Click(object sender, EventArgs e)
        {
            Viewer.Images.Remove(GetSelectedImage());
            ActiveImage = null;
        }

        private void DeleteAllToolStripMenuItem_Click(object sender, EventArgs e)
        {
            Viewer.Images.Clear();
        }

        private void VerifyCECHOToolStripMenuItem_Click(object sender, EventArgs e)
        {
            int result = DicomGlobal.Echo(OptionsBox.StorageNode.Text, Convert.ToInt32(OptionsBox.StoragePort.Text), OptionsBox.CallingAE.Text, OptionsBox.StorageAE.Text);
            MessageBox.Show(VerifyCECHOMessage + result.ToString());
        }

        private void QueryRetrieveCFINDCGETCMOVEToolStripMenuItem_Click(object sender, EventArgs e)
        {
            RetrieveBox.LoadPatient(OptionsBox, Viewer);
        }

        private void SendSelectedCSTOREToolStripMenuItem_Click(object sender, EventArgs e)
        {
            if (GetSelectedImage() != null)
            {
                int Status = GetSelectedImage().Send(OptionsBox.StorageNode.Text, Convert.ToInt32(OptionsBox.StoragePort.Text), OptionsBox.CallingAE.Text, OptionsBox.StorageAE.Text);
                MessageBox.Show(SendSelectedCSTOREMessage + Status.ToString());
            }
        }

        private void SendAllCSTOREToolStripMenuItem_Click(object sender, EventArgs e)
        {
            // this is a simple, safe method - one image per association
            // for production use, a single association would be better
            foreach (DicomImage image in Viewer.Images)
            {
                int Status = image.Send(OptionsBox.StorageNode.Text, Convert.ToInt32(OptionsBox.StoragePort.Text), OptionsBox.CallingAE.Text, OptionsBox.StorageAE.Text);
                if (Status != 0)
                    MessageBox.Show(SendAllCSTOREFailedMessage + Status.ToString());
            }
        }

        private void MouseFunctionChanged(object sender, EventArgs e)
        {
            if ((sender as RadioButton).Checked && sender == AddAnnotation)
                LabelForm.ShowDialog();

            string[] text = MouseFunctionNames[MouseFunction];
            LeftFunction.Text = text[0];
            MiddleFunction.Text = text[1];
            RightFunction.Text = text[2];
        }

        private void CropToLastLabelToolStripMenuItem_Click(object sender, EventArgs e)
        {
            var image = GetSelectedImage();
            if (image != null)
            {
                var label = image.Labels.LastOrDefault(l => l.LabelType == LabelType.Rectangle);
                if (label != null)
                {
                    var LabelArea = Rectangle.Round(label.Area);

                    DicomImage CroppedImage = GetSelectedImage().SubImage(LabelArea.Location, LabelArea.Size, 1.0F, 1);

                    Viewer.AutoDisplay = true;
                    Viewer.Images.Add(CroppedImage);
                }
            }
        }

        private void OptionsToolStripMenuItem_Click(object sender, EventArgs e)
        {
            OptionsBox.tabControl1.SelectedTab = OptionsBox.tabControl1.TabPages[3];
            OptionsBox.ShowDialog();
        }

        private void ShowLoggingOptionsToolStripMenuItem_Click(object sender, EventArgs e)
        {
            OptionsBox.tabControl1.SelectedTab = OptionsBox.tabControl1.TabPages[5];
            OptionsBox.ShowDialog();
        }

        private void ShowPatientAndExamDetails_CheckedChanged(object sender, EventArgs e)
        {
            UpdateInformation();
        }

        private void ShowRuler_CheckedChanged(object sender, EventArgs e)
        {
            UpdateRulers();
        }

        private void Load3DToolStripMenuItem_Click(object sender, EventArgs e)
        {
            OpenFileDialog ofd = new OpenFileDialog()
            {
                Title = OpenFileDialogTitleFor3DImages,
                Multiselect = true
            };

            var samplePath = Path.Combine(SampleExtensions.GetSampleImageFolderPath(), SampleExtensions.IMAGE_FOLDER_3D);
            if (Directory.Exists(samplePath))
            {
                ofd.InitialDirectory = samplePath;
            }

            if (ofd.ShowDialog() == DialogResult.OK)
            {
                Cursor = Cursors.WaitCursor;

                DicomImageCollection collection = new DicomImageCollection();
                collection.Read(ofd.FileNames);
                DicomVolume volume = (collection.Count == 1) ? new DicomVolume(collection[0]) : new DicomVolume(collection);

                // this example loads 2 copies, to show:
                // a) That multiple copies can be displayed from one set of data
                // b) How Dynamic reference lines can be used.

                DicomImage3D mpr1 = new DicomImage3D(volume, RenderMode3D.MPR);

                //Enable Directx mode here else we get an error in the viewer as 3D is not supported by GDI rednering.
                SetViewerMode(ViewerMode.DirectX);

                // centred on image, viewing along Y axis, with -ve Z (top of head) upwards
                mpr1.SetViewPlane(Plane.Coronal, true, 1);
                Viewer.Images.Add(mpr1);

                DicomImage3D mpr2 = new DicomImage3D(volume, RenderMode3D.MPR);

                // centred on image, viewing along X axis, with -ve Z (top of head) upwards
                mpr2.SetViewPlane(Plane.Axial, true, 1);
                Viewer.Images.Add(mpr2);

                // Give each a reference line pointing to the other
                mpr1.Labels.Add(new DicomLabel()
                {
                    LabelType = LabelType.ReferenceLine,
                    RefImage = mpr2,
                    Pen = Pens.Red
                });

                mpr2.Labels.Add(new DicomLabel()
                {
                    LabelType = LabelType.ReferenceLine,
                    RefImage = mpr1,
                    Pen = Pens.Yellow
                });

                RotatePanZoom.Checked = true;
                Cursor = Cursors.Default;
                SetSelectedImage(mpr1);
                Viewer.AdjustMultiRowsColumns();
            }
        }

        private void ImageInformationToolStripMenuItem_Click(object sender, EventArgs e)
        {
            if (GetSelectedImage() == null) return;
            AttributeForm.AttributeList.Text = Utilities.DicomToText(GetSelectedImage());
            AttributeForm.ShowDialog();
        }

        private void AnonymiserToolStripMenuItem_Click(object sender, EventArgs e)
        {
            MessageBox.Show(AnonymiserMessageBoxText, AnonymiserMessageBoxTitle, MessageBoxButtons.OK, MessageBoxIcon.Warning);
            OpenFileDialog d = new OpenFileDialog
            {
                Multiselect = true
            };
            if (d.ShowDialog() == DialogResult.OK && d.FileNames.Any())
            {
                ShowPatientAndExamDetails.Checked = false;
                Viewer.Images.Add(Utilities.Anonymise(d.FileNames));
            }
        }

        #region Cinemode Toolbar Events
        private void EnableCinemodeToolbar()
        {
            tlsCineMode.Visible = Viewer.CurrentImage != null && Viewer.CurrentImage.FrameCount > 1;
        }

        private void TbtnStop_Click(object sender, EventArgs e)
        {
            UpdateImage(i => i.CineMode = CineMode.Static);
        }

        private void TbtnForward_Click(object sender, EventArgs e)
        {
            UpdateImage(i => i.CineMode = CineMode.Forward);
        }

        private void TbtnStart_Click(object sender, EventArgs e)
        {
            UpdateImage(i => { i.Frame = 1; });
        }

        private void TbtnEnd_Click(object sender, EventArgs e)
        {
            UpdateImage(i => { i.Frame = i.FrameCount; });
        }

        private void TbtnRepeat_Click(object sender, EventArgs e)
        {
            UpdateImage(i => i.CineMode = CineMode.Repeat);
        }

        private void TbtnReverse_Click(object sender, EventArgs e)
        {
            UpdateImage(i => i.CineMode = CineMode.Reverse);
        }

        private void MakeNewImageToolStripMenuItem_Click(object sender, EventArgs e)
        {
            Viewer.Images.Add(Utilities.MakeNewImage());
        }

        private void TbtnOscillate_Click(object sender, EventArgs e)
        {
            UpdateImage(i => i.CineMode = CineMode.Oscillate);
        }
        #endregion Cinemode Toolbar Events

        #region Viewermode Functions
        private void DirectXToolStripMenuItem_Click(object sender, EventArgs e)
        {
            SetViewerMode(ViewerMode.DirectX);
        }

        private void GDIToolStripMenuItem_Click(object sender, EventArgs e)
        {
            SetViewerMode(ViewerMode.Gdi);
        }

        private void SetViewerMode(ViewerMode mode)
        {
            Viewer.ViewerMode = mode;
            directXToolStripMenuItem.Checked = mode == ViewerMode.DirectX;
            gDIToolStripMenuItem.Checked = mode == ViewerMode.Gdi;
        }
        #endregion Viewermode Functions

        #region 2D Image Ops
        private void BtnFlipH_Click(object sender, EventArgs e)
        {
            UpdateImage(i => i.FlipState = i.FlipState ^ FlipState.Horizontal);
        }

        private volatile bool running = false;
        private void Animate_Click(object sender, EventArgs e)
        {
            if (running) return;

            System.Threading.Tasks.Task t = new System.Threading.Tasks.Task(() =>
            {
                //Exit if we don't have the default 3d views
                //Note this is sensible if the second Cell view is MPR
                if (Viewer.Images.Count < 2 && !(Viewer.Images[1] is DicomImage3D)) return;

                //Start 3D cine playback
                DicomImage3D img2 = Viewer.Images[1] as DicomImage3D;

                var z1 = img2.DistanceToFarVertex(true);
                var z2 = img2.DistanceToFarVertex(false);

                float spacing = Math.Abs(z1 + z2) / 100.0f; // mm

                // move to one end
                img2.TranslateNormal(-z2);

                int NumberOfSlices = (int)((z1 + z2) / spacing);


                for (int i = 0; i < NumberOfSlices; i++)
                {
                    Viewer.Invoke(new Action(() =>
                    {
                        img2.TranslateNormal(spacing);
                        Viewer.SafeUpdate();
                    }));

                    System.Threading.Thread.Sleep(50);
                }

                // reset to where we were if we wish:
                img2.TranslateNormal(z1 - img2.DistanceToFarVertex(true));
                running = false;

            });

            t.Start();


        }

        private void BtnFlipV_Click(object sender, EventArgs e)
        {
            UpdateImage(i => i.FlipState = i.FlipState ^ FlipState.Vertical);
        }

        private void BtnRotateClock_Click(object sender, EventArgs e)
        {
            UpdateImage(i => i.RotateState = i.RotateState + (int)RotateState.Right);
        }

        private void BtnRotateAnti_Click(object sender, EventArgs e)
        {
            UpdateImage(i => i.RotateState = i.RotateState + (int)RotateState.Left);
        }

        private void BtnReset_Click(object sender, EventArgs e)
        {
            UpdateImage(i =>
            {
                i.FlipState = FlipState.Normal;
                i.RotateState = RotateState.Normal;
                i.Angle = 0.0f;
                i.Zoom = 1.0f;
                i.Scroll = PointF.Empty;
            });
        }

        private void SetPixelLabel(Point pos)
        {
            if (PixelValueLabel == null)
            {
                PixelValueLabel = new DicomLabel()
                {
                    LabelType = LabelType.Text,
                    ScaleMode = ScaleMode.Output,
                    TextBrush = (Brush)Brushes.LawnGreen.Clone(),
                };
                Viewer.Labels.Add(PixelValueLabel);
            }
            //Calculate Pixel location on current image 
            if (ActiveImage != null)
            {
                DicomLabel l = new DicomLabel
                {
                    LabelType = LabelType.Rectangle,
                    Area = new RectangleF(pos, new Size(1, 1)),
                };

                //ROI Label
                l.Rescale(Viewer, ActiveIndex, ScaleMode.Image);

                //Get Pixel Coords in active image world space
                Point3D? worldPos = Viewer.Images[ActiveIndex].ScreenToWorld(pos, Viewer);
                PixelValueLabel.Text = $"[X : {worldPos?.X ?? pos.X:0.0##},  Y : {worldPos?.Y ?? pos.Y:0.0##}, Z: {worldPos?.Z ?? 0:0.0##}]\nPixel Value : {l.ROIMean(Viewer.Images[ActiveIndex], true):0.0###}";

            }
            else
            {
                PixelValueLabel.Text = PixelValueLabel_NoImage;
            }
            PixelValueLabel.Area = new RectangleF(pos.X, pos.Y + 30, 1000, 100);
        }

        #endregion 2D Image Ops

        #region 3D Image Ops
        private void Mode3D_Click(object sender, EventArgs e)
        {
            var Image3D = GetSelectedImage() as DicomImage3D;
            if (Image3D != null)
            {
                Image3D.RenderMode = (RenderMode3D)Enum.Parse(typeof(RenderMode3D), (sender as Button).Name);
                Image3D.SlabThickness = Image3D.RenderMode == RenderMode3D.Average ? 5 : 0;
            }
        }

        private void Plane3D_Click(object sender, EventArgs e)
        {
            UpdateImage(i => (i as IProjection)?.SetViewPlane((Plane)Enum.Parse(typeof(Plane), (sender as Button).Name), true, 1));
        }

        #endregion 3D Image Ops

    }
}

