unit QRUnit;

interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, StdCtrls, ComCtrls, ActiveX, DicomObjects8_TLB;

type
  IDData = record
    level: integer;
    PatientID, StudyUID, SeriesUID, InstanceUID: WideString;
    Next: Pointer;
  end;

type
  pIDData = ^IDData;

type
  TQRForm = class(TForm)
    Tree: TTreeView;
    Retrieve: TButton;
    Cancel: TButton;
    procedure FormActivate(Sender: TObject);
    procedure CancelClick(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure TreeExpanded(Sender: TObject; Node: TTreeNode);
    procedure RetrieveClick(Sender: TObject);
  private
    { Private declarations }
  public
    { Public declarations }
  end;

var
  QRForm: TQRForm;
  Base, Current: pIDData;

implementation

uses DelphiViewer;

{$R *.dfm}

var
  Query: DicomQuery;

function Description(study: DicomDataSet): String;
var
  s: String;
  dt: DicomAttribute;
begin
  s := study.StudyDescription;
  if s = '' then
    s := 'STUDY';
  dt := study.Attributes.Item[$0008, $0020];
  if dt.Exists and (VarType(dt.Value) <> varNULL) then
    s := s + ' on ' + VarToStr(dt.Value);

  Description := s;
end;

function NewIDItem(original: pIDData): pIDData;
var
  Item: pIDData;
begin
  New(Item);
  if original <> nil then
    Item^ := original^;

  if Current = nil then
    Base := Item
  else
    Current^.Next := Item;

  Current := Item;
  Item^.Next := nil;
  NewIDItem := Item;
end;

procedure TQRForm.FormActivate(Sender: TObject);
var
  Results: DicomDataSets;
  Result: DicomDataSet;
  i: integer;
  Node: TTreeNode;
  IDs: pIDData;
begin
  Query := CoDicomQuery.Create;
  Query.Node := MainForm.RemoteIP.text;
  Query.Port := StrToInt(MainForm.RemotePort.text);
  Query.CallingAE := MainForm.LocalAET.text;
  Query.CalledAE := MainForm.RemoteAET.text;

  Base := nil;
  Current := nil;
  Tree.Items.Clear;
  Query.PatientID := '';
  // This could and should be extened to include Patient ID
  Query.Name := InputBox('Query Details',
    'Enter Patient Name.  (Leave as blank or * if unknown)', '*');

  if MainForm.StudyRoot.State = cbChecked then
  begin
    Query.Root := 'STUDY';
    Query.level := 'STUDY';
  end
  else
  begin
    Query.Root := 'PATIENT';
    Query.level := 'PATIENT';
  end;

  Results := Query.DoQuery;

  for i := 1 to Results.Count do
  begin
    Result := Results[i];
    IDs := NewIDItem(nil);

    IDs^.PatientID := Result.PatientID;
    if MainForm.StudyRoot.State = cbChecked then
    begin
      IDs^.level := 2;
      IDs^.StudyUID := Result.StudyUID;
      Node := Tree.Items.AddChildObject(nil, Result.Name + ' / ' +
        Description(Result), IDs);
      Tree.Items.AddChild(Node, 'Please wait while series list is retrieved');
    end
    else
    begin
      IDs^.level := 1;
      Node := Tree.Items.AddChildObject(nil, Result.Name, IDs);
      Tree.Items.AddChild(Node, 'Please wait while study list is retrieved');
    end;
  end;
end;

procedure TQRForm.CancelClick(Sender: TObject);
begin
  Close();
end;

procedure TQRForm.FormClose(Sender: TObject; var Action: TCloseAction);
var
  temp: pIDData;
begin
  // cleanup data objects on heap
  while Base <> nil do
  begin
    temp := Base^.Next;
    Dispose(Base);
    Base := temp;
  end;
end;

procedure TQRForm.TreeExpanded(Sender: TObject; Node: TTreeNode);
var
  res: DicomDataSets;
  r: DicomDataSet;
  i: integer;
  s: String;
  IDs, newIDs: pIDData;
  nd1, oldChild: TTreeNode;
begin
  IDs := pIDData(Node.Data);

  if (IDs^.level < 0) then
    exit; // already expanded

  newIDs := NewIDItem(IDs);

  newIDs.level := IDs.level + 1;
  IDs.level := (-IDs.level); // mark as expanded

  case abs(IDs.level) of

    1:
      begin // 'PATIENT'
        Query.level := 'STUDY';
        Query.PatientID := IDs^.PatientID;
        Query.StudyUID := '';
        Query.SeriesUID := '';
        Query.InstanceUID := '';
        res := Query.DoQuery();

        oldChild := Node.getFirstChild();
        Tree.Items.Delete(oldChild);

        for i := 1 to res.Count do
        begin
          r := res.Item[Variant(i)];
          s := Description(r);
          newIDs.StudyUID := r.StudyUID;
          nd1 := Tree.Items.AddChildObject(Node, s, newIDs);
          Tree.Items.AddChild(nd1,
            'Please wait, while the series list is retrieved');
        end;
      end;

    2:
      begin // 'STUDY'
        Query.level := 'SERIES';
        Query.PatientID := IDs.PatientID;
        Query.StudyUID := IDs.StudyUID;
        Query.SeriesUID := '';
        Query.InstanceUID := '';
        res := Query.DoQuery();

        oldChild := Node.getFirstChild();
        Tree.Items.Delete(oldChild);

        for i := 1 to res.Count do
        begin
          r := res.Item[Variant(i)];
          s := r.SeriesDescription;
          if (s = '') then
            s := '**SERIES**';
          newIDs.SeriesUID := r.SeriesUID;
          nd1 := Tree.Items.AddChildObject(Node, s, newIDs);
          Tree.Items.AddChild(nd1,
            'Please wait, while the image list is retrieved');
        end;
      end;

    3:
      begin // 'SERIES'
        Query.level := 'IMAGE';
        Query.PatientID := IDs.PatientID;
        Query.StudyUID := IDs.StudyUID;
        Query.SeriesUID := IDs.SeriesUID;
        Query.InstanceUID := '';
        res := Query.DoQuery();

        oldChild := Node.getFirstChild();
        Tree.Items.Delete(oldChild);

        for i := 1 to res.Count do
        begin
          r := res.Item[Variant(i)];
          s := r.InstanceUID;
          newIDs.InstanceUID := r.InstanceUID;
          Tree.Items.AddChildObject(Node, s, newIDs);
        end;
      end;
  end;
end;

procedure TQRForm.RetrieveClick(Sender: TObject);
var
  res: DicomImages;
  i: integer;
  IDs: pIDData;
  Node: TTreeNode;
  Port, code: integer;
begin
  Node := Tree.Selected;
  IDs := pIDData(Node.Data);
  Val(MainForm.LocalPort.text, Port, code);

  case abs(IDs.level) of
    1:
      begin // ''PATIENT'
        Query.PatientID := IDs.PatientID;
        Query.StudyUID := '';
        Query.SeriesUID := '';
        Query.InstanceUID := '';
        Query.level := 'PATIENT';
      end;

    2:
      begin // ''STUDY'
        Query.PatientID := IDs.PatientID;
        Query.StudyUID := IDs.StudyUID;
        Query.SeriesUID := '';
        Query.InstanceUID := '';
        Query.level := 'STUDY';
      end;

    3:
      begin // ''SERIES'
        Query.PatientID := IDs.PatientID;
        Query.StudyUID := IDs.StudyUID;
        Query.SeriesUID := IDs.SeriesUID;
        Query.InstanceUID := '';
        Query.level := 'SERIES';
      end;

    4:
      begin // ''IMAGE'
        Query.PatientID := IDs.PatientID;
        Query.StudyUID := IDs.StudyUID;
        Query.SeriesUID := IDs.SeriesUID;
        Query.InstanceUID := IDs.InstanceUID;
        Query.level := 'IMAGE';
      end;
  end;

  case MainForm.RetrievalType.ItemIndex of
  // 0:=C-GET, 1:=Sync C-MOVE, 2:=Async C-MOVE

    0:
      begin
        res := Query.GetImages();
      end;

    1:
      begin
        Query.Destination := MainForm.LocalAET.text;
        Query.ReceivePort := Port;

        // MainForm.Viewer.Unlisten(port);
        res := Query.GetUsingMove();
        // MainForm.Viewer.Listen(port);
      end;

    2:
      begin
        Query.Destination := MainForm.LocalAET.text;
        Query.MoveImages();
      end;
  end; // end case

  if MainForm.RetrievalType.ItemIndex <> 2 then
    for i := 1 to res.Count do
      MainForm.Viewer.Images.Add(res.Item[Variant(i)]);

  Close();
end;

end.
