﻿using System;
using DicomObjects;
using DicomObjects.Enums;
using DicomObjects.EventArguments;
using Microsoft.Extensions.Logging;

namespace FrameLevelRetrieveClient
{
    class Program
    {
        private const string Test1Text = "Test 1 - Frame Level C-GET";
        private const string Frame = "FRAME";
        private const string Client = "CLIENT";
        private const string Node = "localhost";
        private const int Port = 11112;
        private const string CallingAE = "client";
        private const string CalledAE = "server";
        private const string InstanceUID = "1.2.840.10008.2.3.4.5.6.7.8";
        private const string FrameLevelCMoveResult = "Frame Level C-MOVE result";
        private const string PressAnyKeyToContinue = "Press any key to continue";
        private const string Test2Text = "Test 2 - Frame Level C-MOVE";
        private const string PressAnyKeyToExit = "Press any key to exit";

        static void Main(string[] args)
        {
            ILoggerFactory loggerFactory = LoggerFactory.Create(config => config.AddConsole());
            ILogger logger = loggerFactory.CreateLogger<Program>();
            
            // basic logging, change level to 0x3F for more detailed logging
            // LogToFile call will produce DicomObjects log files
            DicomGlobal.LogToLogger(logger, 0x7); 

            Console.WriteLine(Test1Text);
            DicomQuery query = new DicomQuery
            {
                Node = Node,
                Port = Port,
                CallingAE = CallingAE,
                CalledAE = CalledAE,
                Root = QueryRoot.CompositeInstance,
                Level = QueryLevel.FRAME
            };

            DicomDataSet queryDataSet = query.QueryDataSet();
            queryDataSet.Add(Keyword.QueryRetrieveLevel, Frame);
            queryDataSet.InstanceUID = InstanceUID; // a random uid

            // use simple frame list to retrieve frames 1, 12 and 30
            queryDataSet.Add(Keyword.SimpleFrameList, new int[] { 1, 12, 30});
            
            var results = query.Get(queryDataSet);
            for (int frame = 0; frame < results.Count; frame++)
            {
                results[frame].Write($"result_{frame}.dcm");
            }

            Console.WriteLine(PressAnyKeyToContinue);
            Console.ReadKey();

            Console.WriteLine(Test2Text);

            // use calculated frame list to retrieve frames 2, 7, 12, 17, 22 and 27
            queryDataSet.Remove(Keyword.SimpleFrameList);
            queryDataSet.Add(Keyword.CalculatedFrameList, new int[] { 2, 30, 5});
            queryDataSet.Add(Keyword.MoveDestination, Client); // server needs to know what 'CLINET' means

            // set up listener for incoming C-STORE as result of C-MOVE request
            DicomServer server = new DicomServer();
            server.Listen(11112);
            server.InstanceReceived += Server_InstanceReceived;
            
            // result contains is the C-MOVE response, not the images
            var result = query.Move(queryDataSet);
            server.UnlistenAll();

            Console.WriteLine($"{FrameLevelCMoveResult} {result[Keyword.Status].Value}");
            Console.WriteLine(PressAnyKeyToExit);
        }

        private static void Server_InstanceReceived(object Sender, InstanceReceivedArgs e)
        {
            // we don't know the actual frame number, use Guid to form the filename
            // use OriginalTS to write out as is, without transcoding
            e.Instance.Write($"FrameLevelCMOVE_{Guid.NewGuid()}.dcm", e.Instance.OriginalTS);
            e.Status = 0; // success acknowledgement
        }
    }
}
