﻿using DicomObjects;
using System;
using Microsoft.Extensions.Logging;
using DicomObjects.Delegates;
using System.IO;
using Security.Core;

namespace Security
{
    class Program
    {
        private const string Node = "localhost";
        private const string CallingAE = "client";
        private const string CalledAE = "server";
        private const int Port = 104;
        private const string EncryptedFileName = "CT.encrypted";
        private const string Password = "password";

        static void Main(string[] args)
        {
            ILoggerFactory loggerFactory = LoggerFactory.Create(config => config.AddConsole());
            var logger = loggerFactory.CreateLogger<Program>();
            DicomGlobal.LogToLogger(logger, 0x7);

            DicomServer server = new DicomServer();
            server.VerifyReceived += (s, ee) => ee.Status = 0; // return status=0 to verify requests
            int port = Port;
            bool windowsSecuritySCP = true; // server uses Windows security
           
            if (windowsSecuritySCP)
            {
                server.Listen(port, MinimalWindowsServer.GetDecryptedStream);
            }
            else 
            {
                server.Listen(port, MinimalBouncyCastleServer.GetDecryptedStream);
            }

            // test 1 - windows scu echo server
            DicomGlobal.Log("===================== test 1 - windows scu echo server =====================");            
            MinimalWindowsClient WindowsClient = new MinimalWindowsClient();
            DoDicomEcho(WindowsClient.TlsStream);
            DicomGlobal.Log("Press any key to continue");
            Console.ReadKey();

            // test 2 - bouncy castle scu echo server
            DicomGlobal.Log("===================== test 2 - bouncy castle scu echo server =====================");            
            MinimalBouncyCastleClient BC_Client = new MinimalBouncyCastleClient();
            DoDicomEcho(BC_Client.TlsStream);
            DicomGlobal.Log("Press any key to continue");
            Console.ReadKey();

            // test 3 - File encryption and decryption
            DicomGlobal.Log("===================== test 3 - File encryption and decryption =====================");            
            DicomDataSet dataset = new DicomDataSet(Properties.Resources.CT);

            using (FileStream file = new FileStream(EncryptedFileName, FileMode.Create))
            using (Stream encryptedStream = BouncyCastleCmsEncryption.Encrypt(file, Password))
            {
                dataset.Write(encryptedStream);
                DicomGlobal.Log($"{EncryptedFileName} saved");
            }
            Console.ReadKey();

            using (FileStream file = new FileStream(EncryptedFileName, FileMode.Open))
            using (Stream decryptedStream = BouncyCastleCmsEncryption.Decrypt(file, Password))
            {
                MemoryStream ms = new MemoryStream();
                decryptedStream.CopyTo(ms);
                ms.Seek(0, SeekOrigin.Begin);
                DicomDataSet ds = new DicomDataSet(ms);
                DicomGlobal.Log($"{EncryptedFileName} read back in. Patient Name={ds.Name}");
            }

            Console.WriteLine("Press any key to exit");
            Console.ReadKey();
        }

        private static void DoDicomEcho(TlsInitiator se)
        {   
            DicomAssociation cn = new DicomAssociation();
            try
            {
                cn.InitiateTls += se;
                cn.Open(Node, Port, CallingAE, CalledAE);
                var result = cn.Echo();
                DicomGlobal.Log($"C-Echo Result : {result}");
                cn.Close();
            }
            catch (Exception ex)
            {
                DicomGlobal.Log(ex.Message);
            }
        }

    }
}
