﻿using System;
using System.Collections.Generic;
using System.Drawing;
using System.IO;
using DicomObjects;
using DicomObjects.Enums;
using DicomObjects.EventArguments;
using DicomObjects.UIDs;
using DicomObjects.Validation;
using Microsoft.Extensions.Logging;
using SkiaSharp;

namespace PrintServer
{
    internal class PrintServer
    {
        private const string Manufacturer = "Medical Connections";
        private const string ManufacturerModelName = "Demo Printer SCP";
        private const string PrinterStatus = "NORMAL";
        private const string DeviceSerialNumber = "serial no 1234";
        private const string PrinterName = "PrinterName";
        private const string DateFormat = "yyyyMMdd";
        private const string TimeFormat = "hhmmss";
        private const string ServerStartingMessage = "Server Started, Listen Port ";
        private const string FontTimesNewRoman = "Times New Roman";
        private const string FontConsolas = "Consolas";
        private const string FontArial = "Arial";
        private const string AssociationRequestFromMessage = "Association Request from";
        private const string Association = "Association";
        private const string ClosedText = "Closed";
        private const string Portrait = "PORTRAIT";
        private const string StartingPrinting = "Starting Printing";
        private const string PrintingFatalError = "Fatal Error! ";
        private const string PrintingFinished = "Printing Finished";
        private const string Page = "Page ";

        public static DicomDataSet printerObject;
        public int Port { get; set; } = 104;

        readonly DicomDataSetCollection BasicAnnotationBoxes = new DicomDataSetCollection();
        readonly List<DisplayFormats> SupportedDisplayFormats = new List<DisplayFormats>();

        readonly DicomServer Server;
        public class PrinterAssociation : DicomAssociation
        {
            public PrinterAssociation()
            {
                DataSets = new DicomDataSetCollection();
                DataSets.Add(printerObject); // enables N-GET of printer object                
            }

            public DicomDataSetCollection DataSets { get; set; }
            public SessionDataSet FilmSession { get; set; }
            public MyPrintDoc myPrinterDoc;            
        }
        private static void DecodeFormat(string format, out int x, out int y)
        {
            int p, p1;
            p = format.IndexOf(@"\");
            p1 = format.IndexOf(",", p + 1);
            x = Convert.ToInt32(format.Substring(p + 1, p1 - p - 1));
            y = Convert.ToInt32(format.Substring(p1 + 1));
        }
        private static void MakePrinterDataset()
        {
            //  Dummy values as placeholder
            DicomDataSet p = new DicomDataSet
            {
                InstanceUID = WellKnownInstances.Printer
            };
            p.Add(Keyword.SOPClassUID, SOPClasses.Printer);
            p.Add(Keyword.Manufacturer, Manufacturer);
            p.Add(Keyword.ManufacturerModelName, ManufacturerModelName);
            p.Add(Keyword.PrinterStatus, PrinterStatus);
            p.Add(Keyword.PrinterStatusInfo, PrinterStatus);
            p.Add(Keyword.DeviceSerialNumber, DeviceSerialNumber);
            p.Add(Keyword.SoftwareVersions, DicomGlobal.AssemblyVersion);
            p.Add(Keyword.PrinterName, PrinterName);
            p.Add(Keyword.DateOfLastCalibration, DateTime.Now.Date.ToString(DateFormat));
            p.Add(Keyword.TimeOfLastCalibration, DateTime.Now.TimeOfDay.ToString(TimeFormat));
            printerObject = p;
        }

        public PrintServer()
        {
            Server = new DicomServer();
        }
        public void Start()
        {
            ILoggerFactory loggerFactory = LoggerFactory.Create(config => config.AddConsole());
            ILogger logger = loggerFactory.CreateLogger<Program>();
            DicomGlobal.LogToLogger(logger, 0x7); // basic logging, change to 0x3F for more detailed logging

            Server.Listen(Port);
            Server.NewAssociationType = typeof(PrinterAssociation);
            DicomGlobal.Log(ServerStartingMessage + Port.ToString());
            Server.AssociationRequest += Server_AssociationRequest;
            Server.AssociationClosed += Server_AssociationClosed;
            Server.NormalisedReceived += Server_NormalisedReceived;
            Server.VerifyReceived += (o, ee) => { ee.Status = 0; };

            MakePrinterDataset();
            InitializeSupportedDisplayFomats();
        }

        public void Stop()
        {
            Server?.UnlistenAll();
        }

        private void InitializeSupportedDisplayFomats()
        {
            //  Defining SCP supported Annotation boxes
            DicomDataSet annotation1 = new DicomDataSet();
            annotation1.Add(Keyword.ReferencedSOPClassUID, SOPClasses.BasicAnnotationBox);
            annotation1.Add(Keyword.ReferencedSOPInstanceUID, DicomGlobal.NewUID());
            BasicAnnotationBoxes.Add(annotation1);

            DicomDataSet annotation2 = new DicomDataSet();
            annotation2.Add(Keyword.ReferencedSOPClassUID, SOPClasses.BasicAnnotationBox);
            annotation2.Add(Keyword.ReferencedSOPInstanceUID, DicomGlobal.NewUID());
            BasicAnnotationBoxes.Add(annotation2);

            //  Defining display formats for each AnnotationID
            SupportedDisplayFormats.Add(new DisplayFormats(1, new RectangleF(10, 10, 100, 100), new Font(FontTimesNewRoman, 10)));
            SupportedDisplayFormats.Add(new DisplayFormats(2, new RectangleF(150, 50, 100, 200), new Font(FontConsolas, 20)));
            SupportedDisplayFormats.Add(new DisplayFormats(3, new RectangleF(80, 150, 100, 200), new Font(FontArial, 8)));
        }

        private void Server_AssociationRequest(object sender, AssociationRequestArgs e)
        {
            DicomGlobal.Log($"{AssociationRequestFromMessage} {e.Association.CallingAET} At {e.Association.RemoteIP}");
            foreach (DicomContext ct in e.Contexts)
            {
                if (ct.AbstractSyntax == MetaSOPClasses.BasicGrayscalePrint)
                {
                    DicomDataSetCollection nullDataSetCollection = new DicomDataSetCollection();
                    var myPrinterDoc = new MyPrintDoc(nullDataSetCollection)
                    {
                        CurrentPageNumber = 0,
                        TotalPageNumber = 0
                    };
                    myPrinterDoc.PrintPage += MyPrinterDoc_PrintPage;
                    myPrinterDoc.EndPrint += MyPrinterDoc_EndPrint;
                    e.Association.Tag = myPrinterDoc;
                }
            }
        }

        private void Server_AssociationClosed(object sender, AssociationClosedArgs e)
        {
            DicomGlobal.Log($"{Association} {e.Association.AssociationNumber} {ClosedText}");
        }

        private void Server_NormalisedReceived(object sender, NormalisedReceivedArgs e)
        {
            PrinterAssociation assoc = (PrinterAssociation)e.RequestAssociation;
            assoc.myPrinterDoc = e.RequestAssociation.Tag as MyPrintDoc;

            DicomDataSetCollection DataSets = assoc.DataSets;
            DicomDataSet filmBox = null;
            DicomDataSetCollection ImageBoxes;
            DicomDataSet imageBox;
            DicomDataSet command = e.Command;

            int operation = Convert.ToInt32(command[Keyword.CommandField].Value);
            string requestedSOPClass = command[Keyword.RequestedSOPClassUID].Value + "";
            string requestedSOPInstanceUID = command[Keyword.RequestedSOPInstanceUID].Value + "";
            string affectedSOPClass = command[Keyword.AffectedSOPClassUID].Value + "";
            string affectedSOPInstanceUID = command[Keyword.AffectedSOPInstanceUID].Value + "";

            switch (operation)
            {
                case 0x110: // N-GET
                    filmBox = DataSets[requestedSOPInstanceUID];
                    e.ResultDataSet = filmBox;
                    e.Status = 0;
                    break;

                case 0x140: // N-CREATE
                    filmBox = e.Request.Clone() as DicomDataSet; // Set Default Values					
                    filmBox.Add(Keyword.SOPClassUID, affectedSOPClass);

                    if (affectedSOPInstanceUID == "")
                        affectedSOPInstanceUID = DicomGlobal.NewUID();
                    filmBox.Add(Keyword.SOPInstanceUID, affectedSOPInstanceUID);

                    // FilmSession
                    if (affectedSOPClass == SOPClasses.BasicFilmSession)
                    {
                        assoc.FilmSession = new SessionDataSet();
                        foreach (DicomAttribute attr in filmBox)
                        {
                            assoc.FilmSession.Add(attr.Group, attr.Element, attr.Value);
                        }
                        assoc.FilmSession.ValidationOptions = ValidationOptions.None;
                        DataSets.Add(assoc.FilmSession);
                    }

                    // Specific to FilmBox
                    if (affectedSOPClass == SOPClasses.BasicFilmBox)
                    {
                        int cols, rows;
                        DecodeFormat(filmBox[Keyword.ImageDisplayFormat].Value.ToString(), out cols, out rows);
                        ImageBoxes = new DicomDataSetCollection();

                        // e.PresentationContextId is only available in DicomObjects Version 5.7 and onwards
                        bool color = e.RequestAssociation.AgreedContexts[e.PresentationContextId].AbstractSyntax == MetaSOPClasses.BasicColorPrint.ToString();

                        for (int i = 1; i <= rows * cols; i++)
                        {
                            imageBox = new DicomDataSet
                            {
                                InstanceUID = DicomGlobal.NewUID()
                            };
                            if (!color)
                                imageBox.Add(Keyword.ReferencedSOPClassUID, SOPClasses.BasicGrayscaleImageBox);
                            else
                                imageBox.Add(Keyword.ReferencedSOPClassUID, SOPClasses.BasicColorImageBox);

                            imageBox.Add(Keyword.ReferencedSOPInstanceUID, imageBox.InstanceUID);
                            DataSets.Add(imageBox);

                            ImageBoxes.Add(imageBox);
                        }

                        filmBox.Add(Keyword.ReferencedImageBoxSequence, ImageBoxes); // Referenced Image Box Sequence

                        //  Check if SOP class has been proposed, only then add it.
                        foreach (var agreedContext in e.RequestAssociation.AgreedContexts)
                            if (agreedContext.AbstractSyntax == SOPClasses.BasicAnnotationBox.ToString())
                                filmBox.Add(Keyword.ReferencedBasicAnnotationBoxSequence, BasicAnnotationBoxes);

                        // Also Add FilmBox to current session
                        assoc.FilmSession.FilmBoxes.Add(filmBox);

                        DataSets.Add(filmBox);
                    }

                    e.ResultDataSet = filmBox;
                    e.Status = 0;
                    break;

                case 0x120: // N-SET    
                    foreach (DicomDataSet ba in BasicAnnotationBoxes)
                        if (ba[Keyword.ReferencedSOPInstanceUID].Value.ToString() == requestedSOPInstanceUID)
                            filmBox = ba;
                    if (filmBox == null)
                        filmBox = DataSets[requestedSOPInstanceUID];

                    foreach (DicomAttribute a in e.Request)
                    {
                        filmBox.Add(a.Group, a.Element, a.Value);
                    }
                    e.Status = 0;
                    break;

                case 0x130: // N-ACTION					

                    if (requestedSOPClass == SOPClasses.BasicFilmBox)       //  BasicFilmBox  is provided in the request
                    {
                        //  Increment pages
                        assoc.myPrinterDoc.CurrentPageNumber += 1;

                        e.Status = PrintFilmBox(assoc, requestedSOPInstanceUID, DataSets);
                    }
                    else //  We assume that BasicFilmSession is provided instead
                    {
                        int status = 0xC000;

                        //  Iterate and Print each FilmBox
                        foreach (DicomDataSet filmbox in assoc.FilmSession.FilmBoxes)
                        {
                            //  Set total pages since we have the complete session details
                            assoc.myPrinterDoc.TotalPageNumber = assoc.FilmSession.FilmBoxes.Count;
                            //  And print 1 at a time
                            assoc.myPrinterDoc.CurrentPageNumber = 1;

                            status = PrintFilmBox(assoc, filmbox.InstanceUID, DataSets);
                        }
                        e.Status = status;
                    }
                    break;

                case 0x150: //N-Delete
                    DataSets.Remove(requestedSOPInstanceUID);

                    if (requestedSOPClass == SOPClasses.BasicFilmBox)
                        assoc.FilmSession.FilmBoxes.Remove(requestedSOPInstanceUID);

                    if (requestedSOPClass == SOPClasses.BasicFilmSession)
                        assoc.FilmSession = null;

                    e.Status = 0;
                    break;
            }
        }

        private static int PrintFilmBox(PrinterAssociation assoc, string requestedSOPInstanceUID, DicomDataSetCollection DataSets)
        {
            assoc.myPrinterDoc.UID = requestedSOPInstanceUID;
            assoc.myPrinterDoc.PrintDataSets = DataSets;

            DicomDataSet filmbox = DataSets[requestedSOPInstanceUID];
            string orientation = filmbox[Keyword.FilmOrientation].Value.ToString();
            if (orientation == Portrait)
            {
                assoc.myPrinterDoc.DefaultPageSettings.Landscape = false;
            }
            else
            {
                assoc.myPrinterDoc.DefaultPageSettings.Landscape = true;
            }

            //  Print Document
            assoc.myPrinterDoc.Print();

            //  return Status
            return assoc.myPrinterDoc.Status;
        }

        private void MyPrinterDoc_PrintPage(object sender, System.Drawing.Printing.PrintPageEventArgs e)
        {
            DicomGlobal.Log(StartingPrinting);

            MyPrintDoc mp = (MyPrintDoc)sender;

            DicomDataSetCollection datasets = mp.PrintDataSets;
            DicomDataSet filmbox = datasets[mp.UID];
            DicomDataSetCollection imageBoxes = filmbox[Keyword.ReferencedImageBoxSequence].Value as DicomDataSetCollection;

            DecodeFormat(filmbox[Keyword.ImageDisplayFormat].Value.ToString(), out int columns, out int rows);

            int margin = 5;
            int width, height;
            int paperWidth, paperHeight;

            if (!mp.DefaultPageSettings.Landscape)
            {
                paperWidth = mp.PrinterSettings.PaperSizes[0].Width;
                paperHeight = mp.PrinterSettings.PaperSizes[0].Height;
            }
            else
            {
                paperWidth = mp.PrinterSettings.PaperSizes[0].Height;
                paperHeight = mp.PrinterSettings.PaperSizes[0].Width;
            }

            width = ((paperWidth - margin) / columns) - margin;
            height = ((paperHeight - margin) / rows) - margin;

            Point LeftTop = new Point();
            Font textFont = new Font(FontArial, 8);
            string pageNumber = Page + mp.CurrentPageNumber.ToString();
            e.Graphics.DrawString(pageNumber, textFont, Brushes.Blue, 4, 4);

            for (int row = 0; row < rows; row++)
            {
                for (int column = 0; column < columns; column++)
                {
                    DicomDataSet ds2 = datasets[imageBoxes[row * columns + column][Keyword.ReferencedSOPInstanceUID].Value.ToString()];
                    DicomAttribute imageAtt;

                    // if Mono Image
                    if (ds2[Keyword.BasicGrayscaleImageSequence].Exists)
                    {
                        imageAtt = ds2[Keyword.BasicGrayscaleImageSequence];
                    }
                    // if not mono image, then check for a color image
                    else
                    {
                        imageAtt = ds2[Keyword.BasicColorImageSequence];
                    }

                    if (imageAtt.Exists) // has real image then print
                    {
                        DicomDataSetCollection dss2 = (DicomDataSetCollection)imageAtt.Value;
                        // Get the DicomImage to print
                        DicomDataSet dcmImage = dss2[0];
                        // Set up the Print Area						 
                        int min;
                        int offSetX = 0;
                        int offSetY = 0;
                        if (width > height)
                        {
                            min = height;
                            offSetX = (paperWidth - min * columns - (columns + 1) * margin) / 2;
                        }
                        else
                        {
                            min = width;
                            offSetY = (paperHeight - min * rows - (rows + 1) * margin) / 2;
                        }

                        LeftTop.X = column * (min + margin) + margin + offSetX;
                        LeftTop.Y = row * (min + margin) + margin + offSetY;

                        Rectangle PrintArea = new Rectangle(LeftTop.X, LeftTop.Y, min, min);
                        DicomImage dicomImage = new DicomImage(dcmImage);

                        var printImage = SkBitmapToImage(dicomImage.Bitmap());

                        try
                        {
                            // Print Image here
                            e.Graphics.DrawImage(printImage, PrintArea);
                            mp.Status = 0;
                        }
                        catch (Exception ex)
                        {
                            DicomGlobal.Log(PrintingFatalError + ex.Message);
                        }
                    }
                }
            }

        }


        public static Image SkBitmapToImage(SKBitmap skBitmap)
        {
            using (MemoryStream memoryStream = new MemoryStream())
            {
                SKImage skImage = SKImage.FromBitmap(skBitmap);
                SKData skData = skImage.Encode(SKEncodedImageFormat.Png, 100);

                skData.SaveTo(memoryStream);

                memoryStream.Seek(0, SeekOrigin.Begin);

                return Image.FromStream(memoryStream);
            }
        }

        private void MyPrinterDoc_EndPrint(object sender, System.Drawing.Printing.PrintEventArgs e)
        {
            DicomGlobal.Log(PrintingFinished);
        }
    }
}
