using System.Security.Principal;
using DicomObjects;
using Application = System.Windows.Forms.Application;
using Color = System.Drawing.Color;

namespace LicenceManager
{
    public partial class MainForm : Form
    {
        public MainForm()
        {
            InitializeComponent();
        }

        private void MainForm_Load(object sender, EventArgs e)
        {
            if (!new WindowsPrincipal(WindowsIdentity.GetCurrent()).IsInRole(WindowsBuiltInRole.Administrator))
            {
                MessageBox.Show("Please Run the Licence Manager as Administrator");
                Close();
                Application.Exit();
            }
            else
            {
                Text += $" (Assembly version {DicomGlobal.FileVersion})";
                // this helps trouble-shoot license problems
                DicomGlobal.SetRegWord("LogMachineIDInfo", 1);

                // if there is a problem with the license activation, send us the log for checking
                // you should find the D.O.Net.xxx.log in the LicenceManager.exe folder 
                DicomGlobal.LogToFile(Environment.CurrentDirectory, 0x3F);

                FillBox();
            }
        }

        private void FillBox()
        {
            licenseList.Items.Clear();

            mode.Text = Environment.Is64BitProcess ? "64 bit mode" : "32 bit mode";
            // DicomLicensne.MachineID is generated by DicomObjects to uniquely identifies each machine
            // MachineID might change due to hardware changes, but should not change after Windows update
            machineID.Text = "MachineID = " + DicomLicence.MachineId;

            // list all DicomObjects licenses found on the machine, including expired ones    
            foreach (DicomLicence l in DicomLicence.InstalledLicences)
            {
                var item = new ListViewItem(l.Type.ToString())
                {
                    Tag = l
                };
                item.SubItems.Add(l.Expiry.ToString("D"));
                item.SubItems.Add(l.Source);
                item.ForeColor = (l.Expiry < DateTime.Now) ? Color.Red : Color.Green;
                licenseList.Items.Add(item);
            }
        }

        private void GetLicenseBTN_Click(object sender, EventArgs e)
        {
            LicenceDialog d = new LicenceDialog();
            d.ShowDialog();
            FillBox();
        }

        private void DeleteSelectedBTN_Click(object sender, EventArgs e)
        {
            if (licenseList.SelectedItems.Count > 0)
            {
                try
                {
                    DicomLicence l = licenseList.SelectedItems[0].Tag as DicomLicence;
                    if (l != null)
                    {
                        if (MessageBox.Show("Are you really sure?", "Deactivate Selected?", MessageBoxButtons.YesNo, MessageBoxIcon.Exclamation) == DialogResult.Yes)
                        {
                            l.Deactivate(DicomLicence.MachineId, DicomGlobal.FileVersion);
                            FillBox();
                        }
                    }
                }
                catch (Exception ex)
                {
                    MessageBox.Show("Error Deactivating Selected Licence - " + ex.Message);
                }
            }
        }

        private void DeActivateAllLicenseBTN_Click(object sender, EventArgs e)
        {
            if (licenseList.Items.Count > 0)
            {
                try
                {
                    UserInfoPrompt dlg = new UserInfoPrompt();
                    if (dlg.ShowDialog() == DialogResult.OK
                        && !string.IsNullOrEmpty(dlg.CustomerInfo1)
                        && !string.IsNullOrEmpty(dlg.CustomerInfo2))
                    {
                        DicomLicence.DeactivateAll(dlg.CustomerInfo1, dlg.CustomerInfo2);
                        FillBox();
                    }
                }
                catch (Exception ex)
                {
                    MessageBox.Show("Error Deactivating all Licences - " + ex.Message);
                }
            }
        }
    }
}