﻿using DicomObjects;
using DicomObjects.Enums;
using DicomObjects.UIDs;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Security.Cryptography;
using System.Text;
using System.Threading.Tasks;

namespace MauiSampleViewer
{
    static class Utilities
    {
        private const string PixelDataText = "Pixel data";
        /// <summary>
        /// Make a DICOM Secondary Image from scratch
        /// </summary>
        /// <returns>A New DicomImage Object</returns>
        public static DicomImage MakeNewImage()
        {
            DicomImage Image = new DicomImage();
            int s = 513;
            int r = 256;
            byte[,] Pixel = new byte[s, s];
            int i, j;

            Image.Name = "Test Image";
            Image.InstanceUID = DicomGlobal.NewUID();
            Image.PatientID = "Test Image 001";
            Image.DateOfBirth = DateTime.Now;
            Image.StudyUID = DicomGlobal.NewUID();
            Image.SeriesUID = DicomGlobal.NewUID();
            Image.DataSet.Add(Keyword.Modality, "OT");
            Image.AccessionNumber = "1";

            Image.DataSet.Add(Keyword.SOPClassUID, SOPClasses.SecondaryCapture);
            Image.DataSet.Add(Keyword.SamplesPerPixel, 1);
            Image.DataSet.Add(Keyword.PhotometricInterpretation, "MONOCHROME2");
            Image.DataSet.Add(Keyword.Rows, s);
            Image.DataSet.Add(Keyword.Columns, s);
            Image.DataSet.Add(Keyword.BitsAllocated, 8);
            Image.DataSet.Add(Keyword.BitsStored, 8);
            Image.DataSet.Add(Keyword.HighBit, 7);

            for (i = 1; i < s; i++)
            {
                for (j = 1; j < s; j++)
                {
                    Pixel[i, j] = Convert.ToByte(((i + j) / 2) / 256);
                }
            }

            for (i = -r; i <= r; i++)
            {
                for (j = -r; j <= r; j++)
                {
                    if ((i * i + j * j) < (r * r))
                    {
                        Pixel[i + s / 2, j + s / 2] = Convert.ToByte(Math.Sqrt((i * i + j * j) / ((double)(r * r))) * 255);
                    }
                }
            }

            Image.DataSet.Add(0x7FE0, 0x0010, Pixel);
            return Image;
        }

        /// <summary>
        /// utility routine to return an enum representing the checked box in a group
        /// </summary>
        //public static T GroupBoxToEnum<T>(GroupBox Group)
        //{
        //    var button = Group.Controls.Cast<Control>().FirstOrDefault(x => (x as RadioButton).Checked);
        //    string Name = button.Name.Replace("Button", "");
        //    return (T)Enum.Parse(typeof(T), Name);
        //}

        /// <summary>
        /// Dumps all DICOM Attributes to StringBuilder
        /// </summary>        
        static void AppendAttributes(StringBuilder list, string prefix, DicomDataSet ob)
        {
            DicomDataSetCollection? sequence;
            object v;

            foreach (DicomAttribute at in ob)
            {
                list.Append($"{prefix}({at.Group:X4},{at.Element:X4}) : {at.Description:30}:");
                // pixel data
                if ((at.KeywordCode == Keyword.PixelData))
                {
                    list.AppendLine(PixelDataText);
                }
                else
                {
                    sequence = at.Value as DicomDataSetCollection;
                    if (sequence != null)
                    {
                        AppendSequence(list, sequence, prefix);
                    }
                    else
                    {
                        // could be value or array
                        v = at.Value;
                        // i.e. an array
                        if (v is Array a)
                        {
                            AppendArray(list, a);
                        }
                        else
                        {
                            AppendObject(list, at, v);
                        }
                    }
                }
            }
        }

        #region AppendAttributes - sequence, array, object
        private static void AppendSequence(StringBuilder list, DicomDataSetCollection? sequence, string prefix)
        {
            if (sequence != null)
            {
                list.Append("Sequence of ").Append(sequence.Count).AppendLine(" items:");
                foreach (var seqitem in sequence)
                {
                    list.AppendLine($"{prefix}>---------------");
                    AppendAttributes(list, prefix + ">", seqitem);
                }
            }

            list.AppendLine($"{prefix}>---------------");
        }

        private static void AppendArray(StringBuilder list, Array a)
        {
            list.AppendLine("Multiple values :");
            list.Append("              ");

            if (a.Length > 32)
            {
                list.Append($"Array of {a.Length} elements");
            }
            else
            {
                list.Append(string.Join(", ", a.Cast<object>()));
            }
            list.AppendLine();
        }

        private static void AppendObject(StringBuilder list, DicomAttribute at, object v)
        {
            if (at.VR == "DA" && v is DateTime)
            {
                DateTime d = DateTime.Parse(v.ToString() ?? string.Empty);
                list.AppendLine(d.ToShortDateString());
            }
            else if (at.VR == "TM" && v is DateTime)
            {
                DateTime d = DateTime.Parse(v.ToString() ?? string.Empty);
                list.AppendLine(d.ToShortTimeString());
            }
            else
            {
                list.AppendLine(v?.ToString());
            }
        }

        #endregion

        /// <summary>
        /// String representation of a dataset, for human interpretation
        /// </summary>
        /// <returns></returns>
        public static string DicomToText(DicomImage image)
        {
            StringBuilder strBuilder = new StringBuilder();

            //  Parsing command group attributes
            if (image.DataSet.Command != null)
            {
                AppendAttributes(strBuilder, "", image.DataSet.Command);
            }

            //  Parsing Group 0008 onwards
            AppendAttributes(strBuilder, "", image.DataSet);
            return strBuilder.ToString();
        }

        /// <summary>
        /// Hash Patient Name and Patient ID for anonymisation
        /// </summary>        
        private static string HashNameAndId(string name, string id)
        {
            MD5 md5 = MD5.Create();

            string s = name.Trim() + id.Trim();
            byte[] inputBytes = Encoding.ASCII.GetBytes(s);
            byte[] hash = md5.ComputeHash(inputBytes);

            // step 2, convert byte array to hex string
            StringBuilder sb = new StringBuilder();
            for (int i = 0; i < hash.Length; i++)
                sb.Append(hash[i].ToString("X2"));

            return sb.ToString().Substring(0, 8);
        }

        static readonly List<Keyword> ItemsToAnonymise = new List<Keyword>()
        {
            Keyword.OtherPatientIDs, Keyword.PatientAddress, Keyword.CountryOfResidence,
            Keyword.PatientTelephoneNumbers ,Keyword.MilitaryRank, Keyword.BranchOfService
        };

        /// <summary>
        /// Anonymise a collection of DICOM files
        /// </summary>        
        public static DicomDataSetCollection Anonymise(string[] filesToAnonymise)
        {
            Dictionary<string, string> uidCache = new();
            DicomDataSetCollection results = new DicomDataSetCollection();
            Random random = new Random();
            string accessionNumber = random.Next(1000000).ToString();
            string replacementValue = "Anonymised";

            // offset all dates by fixed amount (this leaves relative dates and ages correct
            TimeSpan dateoffset = new TimeSpan(random.Next(730) - 365, 0, 0, 0);

            foreach (string file in filesToAnonymise)
            {
                if (!DicomGlobal.IsDICOM(file))
                    continue;

                DicomDataSet ds = new DicomDataSet(file);

                string hash = HashNameAndId(ds.Name, ds.PatientID);

                // Remove all private attributes
                ds.RemovePrivateAttributes();

                //Replace UIDs
                ReplaceUiDs(ds, uidCache);

                // Replace all Names
                ReplaceAllNames(ds, replacementValue);

                // Replace all dates
                ReplaceAllDates(ds, dateoffset);

                // Anonymise some Patient Level Attributes
                ds.Add(Keyword.PatientName, hash);
                ds.Add(Keyword.PatientID, hash);
                ds.Add(Keyword.AccessionNumber, accessionNumber);

                foreach (Keyword k in ItemsToAnonymise)
                    if (ds[k].ExistsWithValue)
                        ds.Add(k, replacementValue);

                // Try to add 4 white blocks around the image corners by modifying the pixel data
                // Designed to work with single frame images
                if (ds[Keyword.PixelData].Exists)
                {
                    ushort bits = (ushort)ds[Keyword.BitsAllocated].Value;
                    ushort rows = (ushort)ds[Keyword.Rows].Value;
                    ushort columns = (ushort)ds[Keyword.Columns].Value;

                    // set value to mid-scale
                    int whiteValue = (1 << (bits - 1));
                    float blockSize = 0.2F; // 20% of the image

                    var pixels = ds[Keyword.PixelData].Value;

                    if (pixels is ushort[,,] v)
                        WriteBlockForAllFormats(ds, rows, columns, (ushort)whiteValue, blockSize, v);
                    else if (pixels is byte[,,] v1)
                        WriteBlockForAllFormats(ds, rows, columns, (byte)whiteValue, blockSize, v1);

                    ds.Add(Keyword.PixelData, pixels);
                }
                results.Add(ds);
            }
            return results;
        }

        private static void ReplaceUiDs(DicomDataSet ds, Dictionary<string, string> uidCache)
        {
            // replace all UIDs, except for known UIDs (i.e SOP Class UIDs)
            foreach (var attr in ds.Where(a => a.VR == "UI" && a.ExistsWithValue && !a.Value.ToString()!.StartsWith("1.2.840.10008.")).ToList())
            {
                string uid = attr.Value as string ?? string.Empty;
                if (!uidCache.ContainsKey(uid))
                    uidCache.Add(uid, DicomGlobal.NewUID());

                ds.Add(attr.KeywordCode, uidCache[uid]);
            }
        }

        private static void ReplaceAllNames(DicomDataSet ds, string replacementValue)
        {
            foreach (var attr in ds.Where(a => a.VR == "PN" && a.ExistsWithValue).ToList())
                ds.Add(attr.KeywordCode, replacementValue);
        }

        private static void ReplaceAllDates(DicomDataSet ds, TimeSpan dateoffset)
        {
            foreach (var attr in ds.Where(a => a.VR == "DA" && a.ExistsWithValue).ToList())
                ds.Add(attr.KeywordCode, (attr.Value as DateTime?)?.Add(dateoffset));
        }

        private static void WriteBlockForAllFormats<T>(DicomDataSet ds, int rows, int columns, T whiteValue, float blockSize, T[,,] pixel)
        {
            //colour
            if ((ushort)ds[Keyword.SamplesPerPixel].Value == 3)
            {
                // by colour
                if ((ushort)ds[Keyword.PlanarConfiguration].Value == 0)
                {
                    MakeBlocks(rows, columns * 3, whiteValue, blockSize, pixel, 0);
                }
                else // by plane
                {
                    // like 3 mono images concatenated
                    MakeBlocks(rows, columns, whiteValue, blockSize, pixel, 0);
                    MakeBlocks(rows, columns, whiteValue, blockSize, pixel, rows);
                    MakeBlocks(rows, columns, whiteValue, blockSize, pixel, rows * 2);
                }
            }
            else
            {
                // mono or palette (palette will give artitrary colour, but will still be obscured)
                MakeBlocks(rows, columns, whiteValue, blockSize, pixel, 0);
            }
        }

        private static void MakeBlocks<T>(int rows, int columns, T whiteValue, float blockThickness, T[,,] temp, int rowoffset)
        {
            // top left
            for (int x = 0; x < columns * blockThickness; x++)
                for (int y = 0; y < rows * blockThickness; y++)
                    temp[x, y + rowoffset, 0] = whiteValue;
            //top right
            for (int x = columns - 1; x > columns * (1 - blockThickness); x--)
                for (int y = 0; y < rows * blockThickness; y++)
                    temp[x, y + rowoffset, 0] = whiteValue;
            //bottom left
            for (int x = 0; x < columns * blockThickness; x++)
                for (int y = rows - 1; y > rows * (1 - blockThickness); y--)
                    temp[x, y + rowoffset, 0] = whiteValue;
            // bottom right
            for (int x = columns - 1; x > columns * (1 - blockThickness); x--)
                for (int y = rows - 1; y > rows * (1 - blockThickness); y--)
                    temp[x, y + rowoffset, 0] = whiteValue;
        }

        public static T LayoutToEnum<T>(Layout layout) where T : Enum
        {
            var selected = layout.Children
                .OfType<RadioButton>()
                .FirstOrDefault(rb => rb.IsChecked);

            if (selected == null || string.IsNullOrEmpty(selected.AutomationId))
                throw new InvalidOperationException("Selected RadioButton must have a valid AutomationId matching enum name");

            return (T)Enum.Parse(typeof(T), selected.AutomationId);
        }
    }
}
