﻿using System;
using System.Diagnostics;
using System.Threading.Tasks;
using DicomObjects;
using DicomObjects.DicomWeb;
using DicomObjects.Enums;
using Microsoft.Extensions.Logging;

namespace DICOMWebServer.Core
{
    class Program
    {
        protected Program() { }        

        const string serverURL = "https://localhost:9090";
        const string asyncServerURL = "https://localhost:9091";
        static void Main()
        {
            ILoggerFactory loggerFactory = LoggerFactory.Create(config => config.AddConsole());
            ILogger logger = loggerFactory.CreateLogger<Program>();
            DicomGlobal.LogToLogger(logger, 0x3F);

            DicomWebServer server = new();
            _ = server.Listen(serverURL, "TestCert.pfx", "password", false);
            _ = server.Listen(asyncServerURL, "TestCert.pfx", "password", true);

            // event handlers
            server.QidoReceived += Server_QidoReceived;
            server.StowReceived += Server_StowReceived;
            server.WadoReceived += Server_WadoReceived;            

            server.QidoReceivedAsync += Server_QidoReceivedAsync;
            server.StowReceivedAsync += Server_StowReceivedAsync;
            server.WadoReceivedAsync += Server_WadoReceivedAsync;

            Console.WriteLine("Press any key to exit");
            Console.ReadLine();
        }

        private static void Server_WadoReceived(WadoEventArgs args)
        {
            Console.WriteLine("============Server_WadoReceived event============");
            HandleWadoRequest(args).Wait();
        }

        private static Task Server_WadoReceivedAsync(WadoEventArgs args)
        {
            Console.WriteLine("============Server_WadoReceivedAsync event============");
            return HandleWadoRequest(args);
        }

        private static Task HandleWadoRequest(WadoEventArgs args)
        {
            var headers = args.Headers;
            Console.WriteLine("============WadoReceived Headers============");
            foreach (var h in headers)
            {
                Console.WriteLine($"key={h.Key}, value={h.Value}");
            }

            bool useFrameLevel = true; // this depends on the server support

            DicomDataSetCollection results;
            DicomQuery q = args.MakeQuery(useFrameLevel);

            q.Node = "dicomserver.co.uk";
            q.Port = 104;
            q.CalledAE = "wadoClient";
            q.CallingAE = "dicomserver";

            results = q.Get();
            args.ReturnData(results, useFrameLevel);

            return Task.CompletedTask;
        }

        private static void Server_StowReceived(StowEventArgs args)
        {
            Console.WriteLine("============Server_StowReceived event============");
            HandleStowRequest(args).Wait();
        }

        private static Task Server_StowReceivedAsync(StowEventArgs args)
        {
            Console.WriteLine("============Server_StowReceivedAsync event============");
            return HandleStowRequest(args);
        }

        private static Task HandleStowRequest(StowEventArgs args)
        {
            var headers = args.Headers;
            Console.WriteLine("  STOW request Headers  ");
            foreach (var h in headers)
            {
                Console.WriteLine($"    key={h.Key}, value={h.Value}");
            }

            foreach (var ds in args.DataSets)
            {
                int status = 0; //  ds.Send("dicomserver.co.uk", 104, "stowClient", "dicomserver", new string[] { ds.OriginalTS });
                if (status == 0)
                {
                    args.SetSuccess(ds);
                }
                else
                {
                    args.SetFailed(ds, status);
                }
            }

            return Task.CompletedTask;
        }

        private static void Server_QidoReceived(QidoEventArgs args)
        {
            Console.WriteLine("============Server_QidoReceived event============");
            HandleQidoRequest(args).Wait();
        }

        private static Task Server_QidoReceivedAsync(QidoEventArgs args)
        {
            Console.WriteLine("============Server_QidoReceivedAsync event============");
            return HandleQidoRequest(args);
        }

        private static Task HandleQidoRequest(QidoEventArgs args)
        {
            var headers = args.Headers;
            Console.WriteLine("============QIDOReceived Headers============");
            foreach (var h in headers)
            {
                Console.WriteLine($"key={h.Key}, value={h.Value}");
            }

            DicomQuery q = new()
            {
                Node = "dicomserver.co.uk",
                Port = 104,
                CallingAE = "qidoClient",
                CalledAE = "DicomServer",

                Root = QueryRoot.Study,
                Level = args.Level
            };

            // do DICOM C-FIND
            args.ReturnData(q.Find(args.QueryDataSet()));

            if (args.FuzzyMatching)
                args.Warnings.Add("The fuzzymatching parameter is not supported. Only literal matching has been performed.");
            
            return Task.CompletedTask;
        }
    }
}
